/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
'use strict';

var Prism = require('prismjs');
require('prismjs/components/prism-clike');
require('prismjs/components/prism-javascript');
require('prismjs/components/prism-markup');
require('prismjs/components/prism-markdown');
require('prismjs/components/prism-c');
require('prismjs/components/prism-css');
require('prismjs/components/prism-objectivec');
require('prismjs/components/prism-sql');
require('prismjs/components/prism-python');
require('prismjs/components/prism-rust');
require('prismjs/components/prism-swift');
require('prismjs/components/prism-typescript');
var utils = require('@lexical/utils');
var lexical = require('lexical');

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
const DEFAULT_CODE_LANGUAGE = 'javascript';
const CODE_LANGUAGE_FRIENDLY_NAME_MAP = {
  c: 'C',
  clike: 'C-like',
  css: 'CSS',
  html: 'HTML',
  js: 'JavaScript',
  markdown: 'Markdown',
  objc: 'Objective-C',
  plain: 'Plain Text',
  py: 'Python',
  rust: 'Rust',
  sql: 'SQL',
  swift: 'Swift',
  typescript: 'TypeScript',
  xml: 'XML'
};
const CODE_LANGUAGE_MAP = {
  javascript: 'js',
  md: 'markdown',
  plaintext: 'plain',
  python: 'py',
  text: 'plain',
  ts: 'typescript'
};
function normalizeCodeLang(lang) {
  return CODE_LANGUAGE_MAP[lang] || lang;
}
function getLanguageFriendlyName(lang) {
  const _lang = normalizeCodeLang(lang);

  return CODE_LANGUAGE_FRIENDLY_NAME_MAP[_lang] || _lang;
}
const getDefaultCodeLanguage = () => DEFAULT_CODE_LANGUAGE;
const getCodeLanguages = () => Object.keys(Prism.languages).filter( // Prism has several language helpers mixed into languages object
// so filtering them out here to get langs list
language => typeof Prism.languages[language] !== 'function').sort();
/** @noInheritDoc */

class CodeHighlightNode extends lexical.TextNode {
  /** @internal */
  constructor(text, highlightType, key) {
    super(text, key);
    this.__highlightType = highlightType;
  }

  static getType() {
    return 'code-highlight';
  }

  static clone(node) {
    return new CodeHighlightNode(node.__text, node.__highlightType || undefined, node.__key);
  }

  getHighlightType() {
    const self = this.getLatest();
    return self.__highlightType;
  }

  createDOM(config) {
    const element = super.createDOM(config);
    const className = getHighlightThemeClass(config.theme, this.__highlightType);
    utils.addClassNamesToElement(element, className);
    return element;
  }

  updateDOM(prevNode, dom, config) {
    const update = super.updateDOM(prevNode, dom, config);
    const prevClassName = getHighlightThemeClass(config.theme, prevNode.__highlightType);
    const nextClassName = getHighlightThemeClass(config.theme, this.__highlightType);

    if (prevClassName !== nextClassName) {
      if (prevClassName) {
        utils.removeClassNamesFromElement(dom, prevClassName);
      }

      if (nextClassName) {
        utils.addClassNamesToElement(dom, nextClassName);
      }
    }

    return update;
  }

  static importJSON(serializedNode) {
    const node = $createCodeHighlightNode(serializedNode.text, serializedNode.highlightType);
    node.setFormat(serializedNode.format);
    node.setDetail(serializedNode.detail);
    node.setMode(serializedNode.mode);
    node.setStyle(serializedNode.style);
    return node;
  }

  exportJSON() {
    return { ...super.exportJSON(),
      highlightType: this.getHighlightType(),
      type: 'code-highlight',
      version: 1
    };
  } // Prevent formatting (bold, underline, etc)


  setFormat(format) {
    return this;
  }

}

function getHighlightThemeClass(theme, highlightType) {
  return highlightType && theme && theme.codeHighlight && theme.codeHighlight[highlightType];
}

function $createCodeHighlightNode(text, highlightType) {
  return lexical.$applyNodeReplacement(new CodeHighlightNode(text, highlightType));
}
function $isCodeHighlightNode(node) {
  return node instanceof CodeHighlightNode;
}
function getFirstCodeHighlightNodeOfLine(anchor) {
  let currentNode = null;
  const previousSiblings = anchor.getPreviousSiblings();
  previousSiblings.push(anchor);

  while (previousSiblings.length > 0) {
    const node = previousSiblings.pop();

    if ($isCodeHighlightNode(node)) {
      currentNode = node;
    }

    if (lexical.$isLineBreakNode(node)) {
      break;
    }
  }

  return currentNode;
}
function getLastCodeHighlightNodeOfLine(anchor) {
  let currentNode = null;
  const nextSiblings = anchor.getNextSiblings();
  nextSiblings.unshift(anchor);

  while (nextSiblings.length > 0) {
    const node = nextSiblings.shift();

    if ($isCodeHighlightNode(node)) {
      currentNode = node;
    }

    if (lexical.$isLineBreakNode(node)) {
      break;
    }
  }

  return currentNode;
}

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */

const mapToPrismLanguage = language => {
  // eslint-disable-next-line no-prototype-builtins
  return language != null && Prism.languages.hasOwnProperty(language) ? language : undefined;
};

function hasChildDOMNodeTag(node, tagName) {
  for (const child of node.childNodes) {
    if (child instanceof HTMLElement && child.tagName === tagName) {
      return true;
    }

    hasChildDOMNodeTag(child, tagName);
  }

  return false;
}

const LANGUAGE_DATA_ATTRIBUTE = 'data-highlight-language';
/** @noInheritDoc */

class CodeNode extends lexical.ElementNode {
  /** @internal */
  static getType() {
    return 'code';
  }

  static clone(node) {
    return new CodeNode(node.__language, node.__key);
  }

  constructor(language, key) {
    super(key);
    this.__language = mapToPrismLanguage(language);
  } // View


  createDOM(config) {
    const element = document.createElement('code');
    utils.addClassNamesToElement(element, config.theme.code);
    element.setAttribute('spellcheck', 'false');
    const language = this.getLanguage();

    if (language) {
      element.setAttribute(LANGUAGE_DATA_ATTRIBUTE, language);
    }

    return element;
  }

  updateDOM(prevNode, dom) {
    const language = this.__language;
    const prevLanguage = prevNode.__language;

    if (language) {
      if (language !== prevLanguage) {
        dom.setAttribute(LANGUAGE_DATA_ATTRIBUTE, language);
      }
    } else if (prevLanguage) {
      dom.removeAttribute(LANGUAGE_DATA_ATTRIBUTE);
    }

    return false;
  }

  static importDOM() {
    return {
      // Typically <pre> is used for code blocks, and <code> for inline code styles
      // but if it's a multi line <code> we'll create a block. Pass through to
      // inline format handled by TextNode otherwise
      code: node => {
        const isMultiLine = node.textContent != null && (/\r?\n/.test(node.textContent) || hasChildDOMNodeTag(node, 'BR'));
        return isMultiLine ? {
          conversion: convertPreElement,
          priority: 1
        } : null;
      },
      div: node => ({
        conversion: convertDivElement,
        priority: 1
      }),
      pre: node => ({
        conversion: convertPreElement,
        priority: 0
      }),
      table: node => {
        const table = node; // domNode is a <table> since we matched it by nodeName

        if (isGitHubCodeTable(table)) {
          return {
            conversion: convertTableElement,
            priority: 4
          };
        }

        return null;
      },
      td: node => {
        // element is a <td> since we matched it by nodeName
        const td = node;
        const table = td.closest('table');

        if (isGitHubCodeCell(td)) {
          return {
            conversion: convertTableCellElement,
            priority: 4
          };
        }

        if (table && isGitHubCodeTable(table)) {
          // Return a no-op if it's a table cell in a code table, but not a code line.
          // Otherwise it'll fall back to the T
          return {
            conversion: convertCodeNoop,
            priority: 4
          };
        }

        return null;
      },
      tr: node => {
        // element is a <tr> since we matched it by nodeName
        const tr = node;
        const table = tr.closest('table');

        if (table && isGitHubCodeTable(table)) {
          return {
            conversion: convertCodeNoop,
            priority: 4
          };
        }

        return null;
      }
    };
  }

  static importJSON(serializedNode) {
    const node = $createCodeNode(serializedNode.language);
    node.setFormat(serializedNode.format);
    node.setIndent(serializedNode.indent);
    node.setDirection(serializedNode.direction);
    return node;
  }

  exportJSON() {
    return { ...super.exportJSON(),
      language: this.getLanguage(),
      type: 'code',
      version: 1
    };
  } // Mutation


  insertNewAfter(selection, restoreSelection = true) {
    const children = this.getChildren();
    const childrenLength = children.length;

    if (childrenLength >= 2 && children[childrenLength - 1].getTextContent() === '\n' && children[childrenLength - 2].getTextContent() === '\n' && selection.isCollapsed() && selection.anchor.key === this.__key && selection.anchor.offset === childrenLength) {
      children[childrenLength - 1].remove();
      children[childrenLength - 2].remove();
      const newElement = lexical.$createParagraphNode();
      this.insertAfter(newElement, restoreSelection);
      return newElement;
    } // If the selection is within the codeblock, find all leading tabs and
    // spaces of the current line. Create a new line that has all those
    // tabs and spaces, such that leading indentation is preserved.


    const anchor = selection.anchor.getNode();
    const firstNode = getFirstCodeHighlightNodeOfLine(anchor);

    if (firstNode != null) {
      let leadingWhitespace = 0;
      const firstNodeText = firstNode.getTextContent();

      while (leadingWhitespace < firstNodeText.length && /[\t ]/.test(firstNodeText[leadingWhitespace])) {
        leadingWhitespace += 1;
      }

      if (leadingWhitespace > 0) {
        const whitespace = firstNodeText.substring(0, leadingWhitespace);
        const indentedChild = $createCodeHighlightNode(whitespace);
        anchor.insertAfter(indentedChild);
        selection.insertNodes([lexical.$createLineBreakNode()]);
        indentedChild.select();
        return indentedChild;
      }
    }

    return null;
  }

  canInsertTab() {
    const selection = lexical.$getSelection();

    if (!lexical.$isRangeSelection(selection) || !selection.isCollapsed()) {
      return false;
    }

    return true;
  }

  canIndent() {
    return false;
  }

  collapseAtStart() {
    const paragraph = lexical.$createParagraphNode();
    const children = this.getChildren();
    children.forEach(child => paragraph.append(child));
    this.replace(paragraph);
    return true;
  }

  setLanguage(language) {
    const writable = this.getWritable();
    writable.__language = mapToPrismLanguage(language);
  }

  getLanguage() {
    return this.getLatest().__language;
  }

}
function $createCodeNode(language) {
  return lexical.$applyNodeReplacement(new CodeNode(language));
}
function $isCodeNode(node) {
  return node instanceof CodeNode;
}

function convertPreElement(domNode) {
  return {
    node: $createCodeNode(),
    preformatted: true
  };
}

function convertDivElement(domNode) {
  // domNode is a <div> since we matched it by nodeName
  const div = domNode;
  const isCode = isCodeElement(div);

  if (!isCode && !isCodeChildElement(div)) {
    return {
      node: null
    };
  }

  return {
    after: childLexicalNodes => {
      const domParent = domNode.parentNode;

      if (domParent != null && domNode !== domParent.lastChild) {
        childLexicalNodes.push(lexical.$createLineBreakNode());
      }

      return childLexicalNodes;
    },
    node: isCode ? $createCodeNode() : null,
    preformatted: isCode
  };
}

function convertTableElement() {
  return {
    node: $createCodeNode(),
    preformatted: true
  };
}

function convertCodeNoop() {
  return {
    node: null
  };
}

function convertTableCellElement(domNode) {
  // domNode is a <td> since we matched it by nodeName
  const cell = domNode;
  return {
    after: childLexicalNodes => {
      if (cell.parentNode && cell.parentNode.nextSibling) {
        // Append newline between code lines
        childLexicalNodes.push(lexical.$createLineBreakNode());
      }

      return childLexicalNodes;
    },
    node: null
  };
}

function isCodeElement(div) {
  return div.style.fontFamily.match('monospace') !== null;
}

function isCodeChildElement(node) {
  let parent = node.parentElement;

  while (parent !== null) {
    if (isCodeElement(parent)) {
      return true;
    }

    parent = parent.parentElement;
  }

  return false;
}

function isGitHubCodeCell(cell) {
  return cell.classList.contains('js-file-line');
}

function isGitHubCodeTable(table) {
  return table.classList.contains('js-file-line-container');
}

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
const PrismTokenizer = {
  defaultLanguage: DEFAULT_CODE_LANGUAGE,

  tokenize(code, language) {
    return Prism.tokenize(code, Prism.languages[language || ''] || Prism.languages[this.defaultLanguage]);
  }

};

function isSpaceOrTabChar(char) {
  return char === ' ' || char === '\t';
}

function findFirstNotSpaceOrTabCharAtText(text, isForward) {
  const length = text.length;
  let offset = -1;

  if (isForward) {
    for (let i = 0; i < length; i++) {
      const char = text[i];

      if (!isSpaceOrTabChar(char)) {
        offset = i;
        break;
      }
    }
  } else {
    for (let i = length - 1; i > -1; i--) {
      const char = text[i];

      if (!isSpaceOrTabChar(char)) {
        offset = i;
        break;
      }
    }
  }

  return offset;
}

function getStartOfCodeInLine(anchor) {
  let currentNode = null;
  let currentNodeOffset = -1;
  const previousSiblings = anchor.getPreviousSiblings();
  previousSiblings.push(anchor);

  while (previousSiblings.length > 0) {
    const node = previousSiblings.pop();

    if ($isCodeHighlightNode(node)) {
      const text = node.getTextContent();
      const offset = findFirstNotSpaceOrTabCharAtText(text, true);

      if (offset !== -1) {
        currentNode = node;
        currentNodeOffset = offset;
      }
    }

    if (lexical.$isLineBreakNode(node)) {
      break;
    }
  }

  if (currentNode === null) {
    const nextSiblings = anchor.getNextSiblings();

    while (nextSiblings.length > 0) {
      const node = nextSiblings.shift();

      if ($isCodeHighlightNode(node)) {
        const text = node.getTextContent();
        const offset = findFirstNotSpaceOrTabCharAtText(text, true);

        if (offset !== -1) {
          currentNode = node;
          currentNodeOffset = offset;
          break;
        }
      }

      if (lexical.$isLineBreakNode(node)) {
        break;
      }
    }
  }

  return {
    node: currentNode,
    offset: currentNodeOffset
  };
}
function getEndOfCodeInLine(anchor) {
  let currentNode = null;
  let currentNodeOffset = -1;
  const nextSiblings = anchor.getNextSiblings();
  nextSiblings.unshift(anchor);

  while (nextSiblings.length > 0) {
    const node = nextSiblings.shift();

    if ($isCodeHighlightNode(node)) {
      const text = node.getTextContent();
      const offset = findFirstNotSpaceOrTabCharAtText(text, false);

      if (offset !== -1) {
        currentNode = node;
        currentNodeOffset = offset + 1;
      }
    }

    if (lexical.$isLineBreakNode(node)) {
      break;
    }
  }

  if (currentNode === null) {
    const previousSiblings = anchor.getPreviousSiblings();

    while (previousSiblings.length > 0) {
      const node = previousSiblings.pop();

      if ($isCodeHighlightNode(node)) {
        const text = node.getTextContent();
        const offset = findFirstNotSpaceOrTabCharAtText(text, false);

        if (offset !== -1) {
          currentNode = node;
          currentNodeOffset = offset + 1;
          break;
        }
      }

      if (lexical.$isLineBreakNode(node)) {
        break;
      }
    }
  }

  return {
    node: currentNode,
    offset: currentNodeOffset
  };
}

function textNodeTransform(node, editor, tokenizer) {
  // Since CodeNode has flat children structure we only need to check
  // if node's parent is a code node and run highlighting if so
  const parentNode = node.getParent();

  if ($isCodeNode(parentNode)) {
    codeNodeTransform(parentNode, editor, tokenizer);
  } else if ($isCodeHighlightNode(node)) {
    // When code block converted into paragraph or other element
    // code highlight nodes converted back to normal text
    node.replace(lexical.$createTextNode(node.__text));
  }
}

function updateCodeGutter(node, editor) {
  const codeElement = editor.getElementByKey(node.getKey());

  if (codeElement === null) {
    return;
  }

  const children = node.getChildren();
  const childrenLength = children.length; // @ts-ignore: internal field

  if (childrenLength === codeElement.__cachedChildrenLength) {
    // Avoid updating the attribute if the children length hasn't changed.
    return;
  } // @ts-ignore:: internal field


  codeElement.__cachedChildrenLength = childrenLength;
  let gutter = '1';
  let count = 1;

  for (let i = 0; i < childrenLength; i++) {
    if (lexical.$isLineBreakNode(children[i])) {
      gutter += '\n' + ++count;
    }
  }

  codeElement.setAttribute('data-gutter', gutter);
} // Using `skipTransforms` to prevent extra transforms since reformatting the code
// will not affect code block content itself.
//
// Using extra cache (`nodesCurrentlyHighlighting`) since both CodeNode and CodeHighlightNode
// transforms might be called at the same time (e.g. new CodeHighlight node inserted) and
// in both cases we'll rerun whole reformatting over CodeNode, which is redundant.
// Especially when pasting code into CodeBlock.


const nodesCurrentlyHighlighting = new Set();

function codeNodeTransform(node, editor, tokenizer) {
  const nodeKey = node.getKey();

  if (nodesCurrentlyHighlighting.has(nodeKey)) {
    return;
  }

  nodesCurrentlyHighlighting.add(nodeKey); // When new code block inserted it might not have language selected

  if (node.getLanguage() === undefined) {
    node.setLanguage(tokenizer.defaultLanguage);
  } // Using nested update call to pass `skipTransforms` since we don't want
  // each individual codehighlight node to be transformed again as it's already
  // in its final state


  editor.update(() => {
    updateAndRetainSelection(nodeKey, () => {
      const currentNode = lexical.$getNodeByKey(nodeKey);

      if (!$isCodeNode(currentNode) || !currentNode.isAttached()) {
        return false;
      }

      const code = currentNode.getTextContent();
      const tokens = tokenizer.tokenize(code, currentNode.getLanguage() || tokenizer.defaultLanguage);
      const highlightNodes = getHighlightNodes(tokens);
      const diffRange = getDiffRange(currentNode.getChildren(), highlightNodes);
      const {
        from,
        to,
        nodesForReplacement
      } = diffRange;

      if (from !== to || nodesForReplacement.length) {
        node.splice(from, to - from, nodesForReplacement);
        return true;
      }

      return false;
    });
  }, {
    onUpdate: () => {
      nodesCurrentlyHighlighting.delete(nodeKey);
    },
    skipTransforms: true
  });
}

function getHighlightNodes(tokens) {
  const nodes = [];
  tokens.forEach(token => {
    if (typeof token === 'string') {
      const partials = token.split('\n');

      for (let i = 0; i < partials.length; i++) {
        const text = partials[i];

        if (text.length) {
          nodes.push($createCodeHighlightNode(text));
        }

        if (i < partials.length - 1) {
          nodes.push(lexical.$createLineBreakNode());
        }
      }
    } else {
      const {
        content
      } = token;

      if (typeof content === 'string') {
        nodes.push($createCodeHighlightNode(content, token.type));
      } else if (Array.isArray(content) && content.length === 1 && typeof content[0] === 'string') {
        nodes.push($createCodeHighlightNode(content[0], token.type));
      } else if (Array.isArray(content)) {
        nodes.push(...getHighlightNodes(content));
      }
    }
  });
  return nodes;
} // Wrapping update function into selection retainer, that tries to keep cursor at the same
// position as before.


function updateAndRetainSelection(nodeKey, updateFn) {
  const node = lexical.$getNodeByKey(nodeKey);

  if (!$isCodeNode(node) || !node.isAttached()) {
    return;
  }

  const selection = lexical.$getSelection(); // If it's not range selection (or null selection) there's no need to change it,
  // but we can still run highlighting logic

  if (!lexical.$isRangeSelection(selection)) {
    updateFn();
    return;
  }

  const anchor = selection.anchor;
  const anchorOffset = anchor.offset;
  const isNewLineAnchor = anchor.type === 'element' && lexical.$isLineBreakNode(node.getChildAtIndex(anchor.offset - 1));
  let textOffset = 0; // Calculating previous text offset (all text node prior to anchor + anchor own text offset)

  if (!isNewLineAnchor) {
    const anchorNode = anchor.getNode();
    textOffset = anchorOffset + anchorNode.getPreviousSiblings().reduce((offset, _node) => {
      return offset + _node.getTextContentSize();
    }, 0);
  }

  const hasChanges = updateFn();

  if (!hasChanges) {
    return;
  } // Non-text anchors only happen for line breaks, otherwise
  // selection will be within text node (code highlight node)


  if (isNewLineAnchor) {
    anchor.getNode().select(anchorOffset, anchorOffset);
    return;
  } // If it was non-element anchor then we walk through child nodes
  // and looking for a position of original text offset


  node.getChildren().some(_node => {
    const isText = lexical.$isTextNode(_node);

    if (isText || lexical.$isLineBreakNode(_node)) {
      const textContentSize = _node.getTextContentSize();

      if (isText && textContentSize >= textOffset) {
        _node.select(textOffset, textOffset);

        return true;
      }

      textOffset -= textContentSize;
    }

    return false;
  });
} // Finds minimal diff range between two nodes lists. It returns from/to range boundaries of prevNodes
// that needs to be replaced with `nodes` (subset of nextNodes) to make prevNodes equal to nextNodes.


function getDiffRange(prevNodes, nextNodes) {
  let leadingMatch = 0;

  while (leadingMatch < prevNodes.length) {
    if (!isEqual(prevNodes[leadingMatch], nextNodes[leadingMatch])) {
      break;
    }

    leadingMatch++;
  }

  const prevNodesLength = prevNodes.length;
  const nextNodesLength = nextNodes.length;
  const maxTrailingMatch = Math.min(prevNodesLength, nextNodesLength) - leadingMatch;
  let trailingMatch = 0;

  while (trailingMatch < maxTrailingMatch) {
    trailingMatch++;

    if (!isEqual(prevNodes[prevNodesLength - trailingMatch], nextNodes[nextNodesLength - trailingMatch])) {
      trailingMatch--;
      break;
    }
  }

  const from = leadingMatch;
  const to = prevNodesLength - trailingMatch;
  const nodesForReplacement = nextNodes.slice(leadingMatch, nextNodesLength - trailingMatch);
  return {
    from,
    nodesForReplacement,
    to
  };
}

function isEqual(nodeA, nodeB) {
  // Only checking for code higlight nodes and linebreaks. If it's regular text node
  // returning false so that it's transformed into code highlight node
  if ($isCodeHighlightNode(nodeA) && $isCodeHighlightNode(nodeB)) {
    return nodeA.__text === nodeB.__text && nodeA.__highlightType === nodeB.__highlightType;
  }

  if (lexical.$isLineBreakNode(nodeA) && lexical.$isLineBreakNode(nodeB)) {
    return true;
  }

  return false;
}

function handleMultilineIndent(type) {
  const selection = lexical.$getSelection();

  if (!lexical.$isRangeSelection(selection) || selection.isCollapsed()) {
    return false;
  } // Only run multiline indent logic on selections exclusively composed of code highlights and linebreaks


  const nodes = selection.getNodes();

  for (let i = 0; i < nodes.length; i++) {
    const node = nodes[i];

    if (!$isCodeHighlightNode(node) && !lexical.$isLineBreakNode(node)) {
      return false;
    }
  }

  const startOfLine = getFirstCodeHighlightNodeOfLine(nodes[0]);

  if (startOfLine != null) {
    doIndent(startOfLine, type);
  }

  for (let i = 1; i < nodes.length; i++) {
    const node = nodes[i];

    if (lexical.$isLineBreakNode(nodes[i - 1]) && $isCodeHighlightNode(node)) {
      doIndent(node, type);
    }
  }

  return true;
}

function doIndent(node, type) {
  const text = node.getTextContent();

  if (type === lexical.INDENT_CONTENT_COMMAND) {
    // If the codeblock node doesn't start with whitespace, we don't want to
    // naively prepend a '\t'; Prism will then mangle all of our nodes when
    // it separates the whitespace from the first non-whitespace node. This
    // will lead to selection bugs when indenting lines that previously
    // didn't start with a whitespace character
    if (text.length > 0 && /\s/.test(text[0])) {
      node.setTextContent('\t' + text);
    } else {
      const indentNode = $createCodeHighlightNode('\t');
      node.insertBefore(indentNode);
    }
  } else {
    if (text.indexOf('\t') === 0) {
      // Same as above - if we leave empty text nodes lying around, the resulting
      // selection will be mangled
      if (text.length === 1) {
        node.remove();
      } else {
        node.setTextContent(text.substring(1));
      }
    }
  }
}

function handleShiftLines(type, event) {
  // We only care about the alt+arrow keys
  const selection = lexical.$getSelection();

  if (!lexical.$isRangeSelection(selection)) {
    return false;
  } // I'm not quite sure why, but it seems like calling anchor.getNode() collapses the selection here
  // So first, get the anchor and the focus, then get their nodes


  const {
    anchor,
    focus
  } = selection;
  const anchorOffset = anchor.offset;
  const focusOffset = focus.offset;
  const anchorNode = anchor.getNode();
  const focusNode = focus.getNode();
  const arrowIsUp = type === lexical.KEY_ARROW_UP_COMMAND; // Ensure the selection is within the codeblock

  if (!$isCodeHighlightNode(anchorNode) || !$isCodeHighlightNode(focusNode)) {
    return false;
  }

  if (!event.altKey) {
    // Handle moving selection out of the code block, given there are no
    // sibling thats can natively take the selection.
    if (selection.isCollapsed()) {
      const codeNode = anchorNode.getParentOrThrow();

      if (arrowIsUp && anchorOffset === 0 && anchorNode.getPreviousSibling() === null) {
        const codeNodeSibling = codeNode.getPreviousSibling();

        if (codeNodeSibling === null) {
          codeNode.selectPrevious();
          event.preventDefault();
          return true;
        }
      } else if (!arrowIsUp && anchorOffset === anchorNode.getTextContentSize() && anchorNode.getNextSibling() === null) {
        const codeNodeSibling = codeNode.getNextSibling();

        if (codeNodeSibling === null) {
          codeNode.selectNext();
          event.preventDefault();
          return true;
        }
      }
    }

    return false;
  }

  const start = getFirstCodeHighlightNodeOfLine(anchorNode);
  const end = getLastCodeHighlightNodeOfLine(focusNode);

  if (start == null || end == null) {
    return false;
  }

  const range = start.getNodesBetween(end);

  for (let i = 0; i < range.length; i++) {
    const node = range[i];

    if (!$isCodeHighlightNode(node) && !lexical.$isLineBreakNode(node)) {
      return false;
    }
  } // After this point, we know the selection is within the codeblock. We may not be able to
  // actually move the lines around, but we want to return true either way to prevent
  // the event's default behavior


  event.preventDefault();
  event.stopPropagation(); // required to stop cursor movement under Firefox

  const linebreak = arrowIsUp ? start.getPreviousSibling() : end.getNextSibling();

  if (!lexical.$isLineBreakNode(linebreak)) {
    return true;
  }

  const sibling = arrowIsUp ? linebreak.getPreviousSibling() : linebreak.getNextSibling();

  if (sibling == null) {
    return true;
  }

  const maybeInsertionPoint = arrowIsUp ? getFirstCodeHighlightNodeOfLine(sibling) : getLastCodeHighlightNodeOfLine(sibling);
  let insertionPoint = maybeInsertionPoint != null ? maybeInsertionPoint : sibling;
  linebreak.remove();
  range.forEach(node => node.remove());

  if (type === lexical.KEY_ARROW_UP_COMMAND) {
    range.forEach(node => insertionPoint.insertBefore(node));
    insertionPoint.insertBefore(linebreak);
  } else {
    insertionPoint.insertAfter(linebreak);
    insertionPoint = linebreak;
    range.forEach(node => {
      insertionPoint.insertAfter(node);
      insertionPoint = node;
    });
  }

  selection.setTextNodeRange(anchorNode, anchorOffset, focusNode, focusOffset);
  return true;
}

function handleMoveTo(type, event) {
  const selection = lexical.$getSelection();

  if (!lexical.$isRangeSelection(selection)) {
    return false;
  }

  const {
    anchor,
    focus
  } = selection;
  const anchorNode = anchor.getNode();
  const focusNode = focus.getNode();
  const isMoveToStart = type === lexical.MOVE_TO_START;

  if (!$isCodeHighlightNode(anchorNode) || !$isCodeHighlightNode(focusNode)) {
    return false;
  }

  let node;
  let offset;

  if (isMoveToStart) {
    ({
      node,
      offset
    } = getStartOfCodeInLine(focusNode));
  } else {
    ({
      node,
      offset
    } = getEndOfCodeInLine(focusNode));
  }

  if (node !== null && offset !== -1) {
    selection.setTextNodeRange(node, offset, node, offset);
  }

  event.preventDefault();
  event.stopPropagation();
  return true;
}

function registerCodeHighlighting(editor, tokenizer) {
  if (!editor.hasNodes([CodeNode, CodeHighlightNode])) {
    throw new Error('CodeHighlightPlugin: CodeNode or CodeHighlightNode not registered on editor');
  }

  if (tokenizer == null) {
    tokenizer = PrismTokenizer;
  }

  return utils.mergeRegister(editor.registerMutationListener(CodeNode, mutations => {
    editor.update(() => {
      for (const [key, type] of mutations) {
        if (type !== 'destroyed') {
          const node = lexical.$getNodeByKey(key);

          if (node !== null) {
            updateCodeGutter(node, editor);
          }
        }
      }
    });
  }), editor.registerNodeTransform(CodeNode, node => codeNodeTransform(node, editor, tokenizer)), editor.registerNodeTransform(lexical.TextNode, node => textNodeTransform(node, editor, tokenizer)), editor.registerNodeTransform(CodeHighlightNode, node => textNodeTransform(node, editor, tokenizer)), editor.registerCommand(lexical.INDENT_CONTENT_COMMAND, payload => handleMultilineIndent(lexical.INDENT_CONTENT_COMMAND), lexical.COMMAND_PRIORITY_LOW), editor.registerCommand(lexical.OUTDENT_CONTENT_COMMAND, payload => handleMultilineIndent(lexical.OUTDENT_CONTENT_COMMAND), lexical.COMMAND_PRIORITY_LOW), editor.registerCommand(lexical.KEY_ARROW_UP_COMMAND, payload => handleShiftLines(lexical.KEY_ARROW_UP_COMMAND, payload), lexical.COMMAND_PRIORITY_LOW), editor.registerCommand(lexical.KEY_ARROW_DOWN_COMMAND, payload => handleShiftLines(lexical.KEY_ARROW_DOWN_COMMAND, payload), lexical.COMMAND_PRIORITY_LOW), editor.registerCommand(lexical.MOVE_TO_END, payload => handleMoveTo(lexical.MOVE_TO_END, payload), lexical.COMMAND_PRIORITY_LOW), editor.registerCommand(lexical.MOVE_TO_START, payload => handleMoveTo(lexical.MOVE_TO_START, payload), lexical.COMMAND_PRIORITY_LOW));
}

exports.$createCodeHighlightNode = $createCodeHighlightNode;
exports.$createCodeNode = $createCodeNode;
exports.$isCodeHighlightNode = $isCodeHighlightNode;
exports.$isCodeNode = $isCodeNode;
exports.CODE_LANGUAGE_FRIENDLY_NAME_MAP = CODE_LANGUAGE_FRIENDLY_NAME_MAP;
exports.CODE_LANGUAGE_MAP = CODE_LANGUAGE_MAP;
exports.CodeHighlightNode = CodeHighlightNode;
exports.CodeNode = CodeNode;
exports.DEFAULT_CODE_LANGUAGE = DEFAULT_CODE_LANGUAGE;
exports.PrismTokenizer = PrismTokenizer;
exports.getCodeLanguages = getCodeLanguages;
exports.getDefaultCodeLanguage = getDefaultCodeLanguage;
exports.getEndOfCodeInLine = getEndOfCodeInLine;
exports.getFirstCodeHighlightNodeOfLine = getFirstCodeHighlightNodeOfLine;
exports.getLanguageFriendlyName = getLanguageFriendlyName;
exports.getLastCodeHighlightNodeOfLine = getLastCodeHighlightNodeOfLine;
exports.getStartOfCodeInLine = getStartOfCodeInLine;
exports.normalizeCodeLang = normalizeCodeLang;
exports.registerCodeHighlighting = registerCodeHighlighting;
