/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
'use strict';

var lexical = require('lexical');
var utils = require('@lexical/utils');

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
const TableCellHeaderStates = {
  BOTH: 3,
  COLUMN: 2,
  NO_STATUS: 0,
  ROW: 1
};

/** @noInheritDoc */
class TableCellNode extends lexical.DEPRECATED_GridCellNode {
  /** @internal */

  /** @internal */
  static getType() {
    return 'tablecell';
  }

  static clone(node) {
    return new TableCellNode(node.__headerState, node.__colSpan, node.__width, node.__key);
  }

  static importDOM() {
    return {
      td: node => ({
        conversion: convertTableCellNodeElement,
        priority: 0
      }),
      th: node => ({
        conversion: convertTableCellNodeElement,
        priority: 0
      })
    };
  }

  static importJSON(serializedNode) {
    return $createTableCellNode(serializedNode.headerState, serializedNode.colSpan, serializedNode.width || undefined);
  }

  constructor(headerState = TableCellHeaderStates.NO_STATUS, colSpan = 1, width, key) {
    super(colSpan, key);
    this.__headerState = headerState;
    this.__width = width;
  }

  createDOM(config) {
    const element = document.createElement(this.getTag());

    if (this.__width) {
      element.style.width = `${this.__width}px`;
    }

    utils.addClassNamesToElement(element, config.theme.tableCell, this.hasHeader() && config.theme.tableCellHeader);
    return element;
  }

  exportDOM(editor) {
    const {
      element
    } = super.exportDOM(editor);

    if (element) {
      const maxWidth = 700;
      const colCount = this.getParentOrThrow().getChildrenSize();
      element.style.border = '1px solid black';
      element.style.width = `${this.getWidth() || Math.max(90, maxWidth / colCount)}px`;
      element.style.verticalAlign = 'top';
      element.style.textAlign = 'start';

      if (this.hasHeader()) {
        element.style.backgroundColor = '#f2f3f5';
      }
    }

    return {
      element
    };
  }

  exportJSON() {
    return { ...super.exportJSON(),
      colSpan: super.__colSpan,
      headerState: this.__headerState,
      type: 'tablecell',
      width: this.getWidth()
    };
  }

  getTag() {
    return this.hasHeader() ? 'th' : 'td';
  }

  setHeaderStyles(headerState) {
    const self = this.getWritable();
    self.__headerState = headerState;
    return this.__headerState;
  }

  getHeaderStyles() {
    return this.getLatest().__headerState;
  }

  setWidth(width) {
    const self = this.getWritable();
    self.__width = width;
    return this.__width;
  }

  getWidth() {
    return this.getLatest().__width;
  }

  toggleHeaderStyle(headerStateToToggle) {
    const self = this.getWritable();

    if ((self.__headerState & headerStateToToggle) === headerStateToToggle) {
      self.__headerState -= headerStateToToggle;
    } else {
      self.__headerState += headerStateToToggle;
    }

    return self;
  }

  hasHeaderState(headerState) {
    return (this.getHeaderStyles() & headerState) === headerState;
  }

  hasHeader() {
    return this.getLatest().__headerState !== TableCellHeaderStates.NO_STATUS;
  }

  updateDOM(prevNode) {
    return prevNode.__headerState !== this.__headerState || prevNode.__width !== this.__width;
  }

  isShadowRoot() {
    return true;
  }

  collapseAtStart() {
    return true;
  }

  canBeEmpty() {
    return false;
  }

  canIndent() {
    return false;
  }

}
function convertTableCellNodeElement(domNode) {
  const nodeName = domNode.nodeName.toLowerCase();
  const tableCellNode = $createTableCellNode(nodeName === 'th' ? TableCellHeaderStates.ROW : TableCellHeaderStates.NO_STATUS);
  return {
    forChild: (lexicalNode, parentLexicalNode) => {
      if ($isTableCellNode(parentLexicalNode) && !lexical.$isElementNode(lexicalNode)) {
        const paragraphNode = lexical.$createParagraphNode();

        if (lexical.$isLineBreakNode(lexicalNode) && lexicalNode.getTextContent() === '\n') {
          return null;
        }

        paragraphNode.append(lexicalNode);
        return paragraphNode;
      }

      return lexicalNode;
    },
    node: tableCellNode
  };
}
function $createTableCellNode(headerState, colSpan = 1, width) {
  return lexical.$applyNodeReplacement(new TableCellNode(headerState, colSpan, width));
}
function $isTableCellNode(node) {
  return node instanceof TableCellNode;
}

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */

/** @noInheritDoc */
class TableRowNode extends lexical.DEPRECATED_GridRowNode {
  /** @internal */
  static getType() {
    return 'tablerow';
  }

  static clone(node) {
    return new TableRowNode(node.__height, node.__key);
  }

  static importDOM() {
    return {
      tr: node => ({
        conversion: convertTableRowElement,
        priority: 0
      })
    };
  }

  static importJSON(serializedNode) {
    return $createTableRowNode(serializedNode.height);
  }

  constructor(height, key) {
    super(key);
    this.__height = height;
  }

  exportJSON() {
    return { ...super.exportJSON(),
      type: 'tablerow',
      version: 1
    };
  }

  createDOM(config) {
    const element = document.createElement('tr');

    if (this.__height) {
      element.style.height = `${this.__height}px`;
    }

    utils.addClassNamesToElement(element, config.theme.tableRow);
    return element;
  }

  isShadowRoot() {
    return true;
  }

  setHeight(height) {
    const self = this.getWritable();
    self.__height = height;
    return this.__height;
  }

  getHeight() {
    return this.getLatest().__height;
  }

  updateDOM(prevNode) {
    return prevNode.__height !== this.__height;
  }

  canBeEmpty() {
    return false;
  }

  canIndent() {
    return false;
  }

}
function convertTableRowElement(domNode) {
  return {
    node: $createTableRowNode()
  };
}
function $createTableRowNode(height) {
  return lexical.$applyNodeReplacement(new TableRowNode(height));
}
function $isTableRowNode(node) {
  return node instanceof TableRowNode;
}

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
const CAN_USE_DOM = typeof window !== 'undefined' && typeof window.document !== 'undefined' && typeof window.document.createElement !== 'undefined';

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */

const getDOMSelection = targetWindow => CAN_USE_DOM ? (targetWindow || window).getSelection() : null;

if (CAN_USE_DOM) {
  const disableNativeSelectionUi = document.createElement('style');
  disableNativeSelectionUi.innerHTML = `
    table.disable-selection {
      -webkit-touch-callout: none;
      -webkit-user-select: none; 
      -khtml-user-select: none; 
      -moz-user-select: none; 
      -ms-user-select: none; 
      user-select: none;
    }
  
    .disable-selection span::selection{
      background-color: transparent;
    }
    .disable-selection br::selection{
      background-color: transparent;
    }
  `;

  if (document.body) {
    document.body.append(disableNativeSelectionUi);
  }
}

class TableSelection {
  constructor(editor, tableNodeKey) {
    this.isHighlightingCells = false;
    this.anchorX = -1;
    this.anchorY = -1;
    this.focusX = -1;
    this.focusY = -1;
    this.listenersToRemove = new Set();
    this.tableNodeKey = tableNodeKey;
    this.editor = editor;
    this.grid = {
      cells: [],
      columns: 0,
      rows: 0
    };
    this.gridSelection = null;
    this.anchorCellNodeKey = null;
    this.focusCellNodeKey = null;
    this.anchorCell = null;
    this.focusCell = null;
    this.hasHijackedSelectionStyles = false;
    this.trackTableGrid();
  }

  getGrid() {
    return this.grid;
  }

  removeListeners() {
    Array.from(this.listenersToRemove).forEach(removeListener => removeListener());
  }

  trackTableGrid() {
    const observer = new MutationObserver(records => {
      this.editor.update(() => {
        let gridNeedsRedraw = false;

        for (let i = 0; i < records.length; i++) {
          const record = records[i];
          const target = record.target;
          const nodeName = target.nodeName;

          if (nodeName === 'TABLE' || nodeName === 'TR') {
            gridNeedsRedraw = true;
            break;
          }
        }

        if (!gridNeedsRedraw) {
          return;
        }

        const tableElement = this.editor.getElementByKey(this.tableNodeKey);

        if (!tableElement) {
          throw new Error('Expected to find TableElement in DOM');
        }

        this.grid = getTableGrid(tableElement);
      });
    });
    this.editor.update(() => {
      const tableElement = this.editor.getElementByKey(this.tableNodeKey);

      if (!tableElement) {
        throw new Error('Expected to find TableElement in DOM');
      }

      this.grid = getTableGrid(tableElement);
      observer.observe(tableElement, {
        childList: true,
        subtree: true
      });
    });
  }

  clearHighlight() {
    this.isHighlightingCells = false;
    this.anchorX = -1;
    this.anchorY = -1;
    this.focusX = -1;
    this.focusY = -1;
    this.gridSelection = null;
    this.anchorCellNodeKey = null;
    this.focusCellNodeKey = null;
    this.anchorCell = null;
    this.focusCell = null;
    this.hasHijackedSelectionStyles = false;
    this.enableHighlightStyle();
    this.editor.update(() => {
      const tableNode = lexical.$getNodeByKey(this.tableNodeKey);

      if (!$isTableNode(tableNode)) {
        throw new Error('Expected TableNode.');
      }

      const tableElement = this.editor.getElementByKey(this.tableNodeKey);

      if (!tableElement) {
        throw new Error('Expected to find TableElement in DOM');
      }

      const grid = getTableGrid(tableElement);
      $updateDOMForSelection(grid, null);
      lexical.$setSelection(null);
      this.editor.dispatchCommand(lexical.SELECTION_CHANGE_COMMAND, undefined);
    });
  }

  enableHighlightStyle() {
    this.editor.update(() => {
      const tableElement = this.editor.getElementByKey(this.tableNodeKey);

      if (!tableElement) {
        throw new Error('Expected to find TableElement in DOM');
      }

      tableElement.classList.remove('disable-selection');
      this.hasHijackedSelectionStyles = false;
    });
  }

  disableHighlightStyle() {
    this.editor.update(() => {
      const tableElement = this.editor.getElementByKey(this.tableNodeKey);

      if (!tableElement) {
        throw new Error('Expected to find TableElement in DOM');
      }

      tableElement.classList.add('disable-selection');
      this.hasHijackedSelectionStyles = true;
    });
  }

  updateTableGridSelection(selection) {
    if (selection != null && selection.gridKey === this.tableNodeKey) {
      this.gridSelection = selection;
      this.isHighlightingCells = true;
      this.disableHighlightStyle();
      $updateDOMForSelection(this.grid, this.gridSelection);
    } else if (selection == null) {
      this.clearHighlight();
    }
  }

  setFocusCellForSelection(cell, ignoreStart = false) {
    this.editor.update(() => {
      const tableNode = lexical.$getNodeByKey(this.tableNodeKey);

      if (!$isTableNode(tableNode)) {
        throw new Error('Expected TableNode.');
      }

      const tableElement = this.editor.getElementByKey(this.tableNodeKey);

      if (!tableElement) {
        throw new Error('Expected to find TableElement in DOM');
      }

      const cellX = cell.x;
      const cellY = cell.y;
      this.focusCell = cell;

      if (this.anchorCell !== null) {
        const domSelection = getDOMSelection(this.editor._window); // Collapse the selection

        if (domSelection) {
          domSelection.setBaseAndExtent(this.anchorCell.elem, 0, this.focusCell.elem, 0);
        }
      }

      if (!this.isHighlightingCells && (this.anchorX !== cellX || this.anchorY !== cellY || ignoreStart)) {
        this.isHighlightingCells = true;
        this.disableHighlightStyle();
      } else if (cellX === this.focusX && cellY === this.focusY) {
        return;
      }

      this.focusX = cellX;
      this.focusY = cellY;

      if (this.isHighlightingCells) {
        const focusTableCellNode = lexical.$getNearestNodeFromDOMNode(cell.elem);

        if (this.gridSelection != null && this.anchorCellNodeKey != null && $isTableCellNode(focusTableCellNode)) {
          const focusNodeKey = focusTableCellNode.getKey();
          this.gridSelection = this.gridSelection.clone() || lexical.DEPRECATED_$createGridSelection();
          this.focusCellNodeKey = focusNodeKey;
          this.gridSelection.set(this.tableNodeKey, this.anchorCellNodeKey, this.focusCellNodeKey);
          lexical.$setSelection(this.gridSelection);
          this.editor.dispatchCommand(lexical.SELECTION_CHANGE_COMMAND, undefined);
          $updateDOMForSelection(this.grid, this.gridSelection);
        }
      }
    });
  }

  setAnchorCellForSelection(cell) {
    this.isHighlightingCells = false;
    this.anchorCell = cell;
    this.anchorX = cell.x;
    this.anchorY = cell.y;
    this.editor.update(() => {
      const anchorTableCellNode = lexical.$getNearestNodeFromDOMNode(cell.elem);

      if ($isTableCellNode(anchorTableCellNode)) {
        const anchorNodeKey = anchorTableCellNode.getKey();
        this.gridSelection = lexical.DEPRECATED_$createGridSelection();
        this.anchorCellNodeKey = anchorNodeKey;
      }
    });
  }

  formatCells(type) {
    this.editor.update(() => {
      const selection = lexical.$getSelection();

      if (!lexical.DEPRECATED_$isGridSelection(selection)) {
        {
          throw Error(`Expected grid selection`);
        }
      }

      const formatSelection = lexical.$createRangeSelection();
      const anchor = formatSelection.anchor;
      const focus = formatSelection.focus;
      selection.getNodes().forEach(cellNode => {
        if ($isTableCellNode(cellNode) && cellNode.getTextContentSize() !== 0) {
          anchor.set(cellNode.getKey(), 0, 'element');
          focus.set(cellNode.getKey(), cellNode.getChildrenSize(), 'element');
          formatSelection.formatText(type);
        }
      });
      lexical.$setSelection(selection);
      this.editor.dispatchCommand(lexical.SELECTION_CHANGE_COMMAND, undefined);
    });
  }

  clearText() {
    this.editor.update(() => {
      const tableNode = lexical.$getNodeByKey(this.tableNodeKey);

      if (!$isTableNode(tableNode)) {
        throw new Error('Expected TableNode.');
      }

      const selection = lexical.$getSelection();

      if (!lexical.DEPRECATED_$isGridSelection(selection)) {
        {
          throw Error(`Expected grid selection`);
        }
      }

      const selectedNodes = selection.getNodes().filter($isTableCellNode);

      if (selectedNodes.length === this.grid.columns * this.grid.rows) {
        tableNode.selectPrevious(); // Delete entire table

        tableNode.remove();
        const rootNode = lexical.$getRoot();
        rootNode.selectStart();
        return;
      }

      selectedNodes.forEach(cellNode => {
        if (lexical.$isElementNode(cellNode)) {
          const paragraphNode = lexical.$createParagraphNode();
          const textNode = lexical.$createTextNode();
          paragraphNode.append(textNode);
          cellNode.append(paragraphNode);
          cellNode.getChildren().forEach(child => {
            if (child !== paragraphNode) {
              child.remove();
            }
          });
        }
      });
      $updateDOMForSelection(this.grid, null);
      lexical.$setSelection(null);
      this.editor.dispatchCommand(lexical.SELECTION_CHANGE_COMMAND, undefined);
    });
  }

}

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
const LEXICAL_ELEMENT_KEY = '__lexicalTableSelection';
function applyTableHandlers(tableNode, tableElement, editor) {
  const rootElement = editor.getRootElement();

  if (rootElement === null) {
    throw new Error('No root element.');
  }

  const tableSelection = new TableSelection(editor, tableNode.getKey());
  attachTableSelectionToTableElement(tableElement, tableSelection);
  let isMouseDown = false;
  let isRangeSelectionHijacked = false;
  tableElement.addEventListener('dblclick', event => {
    const cell = getCellFromTarget(event.target);

    if (cell !== null) {
      event.preventDefault();
      event.stopImmediatePropagation();
      event.stopPropagation();
      tableSelection.setAnchorCellForSelection(cell);
      tableSelection.setFocusCellForSelection(cell, true);
      isMouseDown = false;
    }
  }); // This is the anchor of the selection.

  tableElement.addEventListener('mousedown', event => {
    setTimeout(() => {
      if (event.button !== 0) {
        return;
      }

      const cell = getCellFromTarget(event.target);

      if (cell !== null) {
        event.preventDefault();
        event.stopPropagation();
        event.stopImmediatePropagation();
        tableSelection.setAnchorCellForSelection(cell);
      }
    }, 0);
  }); // This is adjusting the focus of the selection.

  tableElement.addEventListener('mousemove', event => {
    if (isRangeSelectionHijacked) {
      event.preventDefault();
      event.stopPropagation();
      event.stopImmediatePropagation();
    }

    if (isMouseDown) {
      const cell = getCellFromTarget(event.target);

      if (cell !== null) {
        const cellX = cell.x;
        const cellY = cell.y;

        if (isMouseDown && (tableSelection.anchorX !== cellX || tableSelection.anchorY !== cellY || tableSelection.isHighlightingCells)) {
          event.preventDefault();
          tableSelection.setFocusCellForSelection(cell);
        }
      }
    }
  }); // Select entire table at this point, when grid selection is ready.

  tableElement.addEventListener('mouseleave', () => {
    if (isMouseDown) {
      return;
    }
  }); // Clear selection when clicking outside of dom.

  const mouseDownCallback = event => {
    isMouseDown = true;

    if (event.button !== 0) {
      return;
    }

    editor.update(() => {
      const selection = lexical.$getSelection();

      if (lexical.DEPRECATED_$isGridSelection(selection) && selection.gridKey === tableSelection.tableNodeKey && rootElement.contains(event.target)) {
        return tableSelection.clearHighlight();
      }
    });
  };

  window.addEventListener('mousedown', mouseDownCallback);
  tableSelection.listenersToRemove.add(() => window.removeEventListener('mousedown', mouseDownCallback));

  const mouseUpCallback = event => {
    if (isMouseDown) {
      event.preventDefault();
      event.stopPropagation();
    }

    isMouseDown = false;
  };

  window.addEventListener('mouseup', mouseUpCallback);
  tableSelection.listenersToRemove.add(() => window.removeEventListener('mouseup', mouseUpCallback));
  tableSelection.listenersToRemove.add(() => tableElement.addEventListener('mouseup', mouseUpCallback));
  tableSelection.listenersToRemove.add(() => tableElement.removeEventListener('mouseup', mouseUpCallback));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.KEY_ARROW_DOWN_COMMAND, event => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    const direction = 'down';

    if (lexical.$isRangeSelection(selection)) {
      if (selection.isCollapsed()) {
        const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

        if (!$isTableCellNode(tableCellNode)) {
          return false;
        }

        const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
        const elementParentNode = utils.$findMatchingParent(selection.anchor.getNode(), n => lexical.$isElementNode(n));

        if (elementParentNode == null) {
          throw new Error('Expected BlockNode Parent');
        }

        const lastChild = tableCellNode.getLastChild();
        const isSelectionInLastBlock = lastChild && elementParentNode.isParentOf(lastChild) || elementParentNode === lastChild;

        if (isSelectionInLastBlock || event.shiftKey) {
          event.preventDefault();
          event.stopImmediatePropagation();
          event.stopPropagation(); // Start Selection

          if (event.shiftKey) {
            tableSelection.setAnchorCellForSelection(tableNode.getCellFromCordsOrThrow(currentCords.x, currentCords.y, tableSelection.grid));
            return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
          }

          return selectGridNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
        }
      }
    } else if (lexical.DEPRECATED_$isGridSelection(selection) && event.shiftKey) {
      const tableCellNode = selection.focus.getNode();

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }

      const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
      event.preventDefault();
      event.stopImmediatePropagation();
      event.stopPropagation();
      return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
    }

    return false;
  }, lexical.COMMAND_PRIORITY_HIGH));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.KEY_ARROW_UP_COMMAND, event => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    const direction = 'up';

    if (lexical.$isRangeSelection(selection)) {
      if (selection.isCollapsed()) {
        const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

        if (!$isTableCellNode(tableCellNode)) {
          return false;
        }

        const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
        const elementParentNode = utils.$findMatchingParent(selection.anchor.getNode(), n => lexical.$isElementNode(n));

        if (elementParentNode == null) {
          throw new Error('Expected BlockNode Parent');
        }

        const lastChild = tableCellNode.getLastChild();
        const isSelectionInLastBlock = lastChild && elementParentNode.isParentOf(lastChild) || elementParentNode === lastChild;

        if (isSelectionInLastBlock || event.shiftKey) {
          event.preventDefault();
          event.stopImmediatePropagation();
          event.stopPropagation(); // Start Selection

          if (event.shiftKey) {
            tableSelection.setAnchorCellForSelection(tableNode.getCellFromCordsOrThrow(currentCords.x, currentCords.y, tableSelection.grid));
            return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
          }

          return selectGridNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
        }
      }
    } else if (lexical.DEPRECATED_$isGridSelection(selection) && event.shiftKey) {
      const tableCellNode = selection.focus.getNode();

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }

      const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
      event.preventDefault();
      event.stopImmediatePropagation();
      event.stopPropagation();
      return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
    }

    return false;
  }, lexical.COMMAND_PRIORITY_HIGH));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.KEY_ARROW_LEFT_COMMAND, event => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    const direction = 'backward';

    if (lexical.$isRangeSelection(selection)) {
      if (selection.isCollapsed()) {
        const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

        if (!$isTableCellNode(tableCellNode)) {
          return false;
        }

        const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
        const elementParentNode = utils.$findMatchingParent(selection.anchor.getNode(), n => lexical.$isElementNode(n));

        if (elementParentNode == null) {
          throw new Error('Expected BlockNode Parent');
        }

        if (selection.anchor.offset === 0 || event.shiftKey) {
          event.preventDefault();
          event.stopImmediatePropagation();
          event.stopPropagation(); // Start Selection

          if (event.shiftKey) {
            tableSelection.setAnchorCellForSelection(tableNode.getCellFromCordsOrThrow(currentCords.x, currentCords.y, tableSelection.grid));
            return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
          }

          return selectGridNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
        }
      }
    } else if (lexical.DEPRECATED_$isGridSelection(selection) && event.shiftKey) {
      const tableCellNode = selection.focus.getNode();

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }

      const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
      event.preventDefault();
      event.stopImmediatePropagation();
      event.stopPropagation();
      return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
    }

    return false;
  }, lexical.COMMAND_PRIORITY_HIGH));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.KEY_ARROW_RIGHT_COMMAND, event => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    const direction = 'forward';

    if (lexical.$isRangeSelection(selection)) {
      if (selection.isCollapsed()) {
        const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

        if (!$isTableCellNode(tableCellNode)) {
          return false;
        }

        const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
        const elementParentNode = utils.$findMatchingParent(selection.anchor.getNode(), n => lexical.$isElementNode(n));

        if (elementParentNode == null) {
          throw new Error('Expected BlockNode Parent');
        }

        if (selection.anchor.offset === selection.anchor.getNode().getTextContentSize() || event.shiftKey) {
          event.preventDefault();
          event.stopImmediatePropagation();
          event.stopPropagation(); // Start Selection

          if (event.shiftKey) {
            tableSelection.setAnchorCellForSelection(tableNode.getCellFromCordsOrThrow(currentCords.x, currentCords.y, tableSelection.grid));
            return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
          }

          return selectGridNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
        }
      }
    } else if (lexical.DEPRECATED_$isGridSelection(selection) && event.shiftKey) {
      const tableCellNode = selection.focus.getNode();

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }

      const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
      event.preventDefault();
      event.stopImmediatePropagation();
      event.stopPropagation();
      return adjustFocusNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, direction);
    }

    return false;
  }, lexical.COMMAND_PRIORITY_HIGH));

  const deleteTextHandler = command => () => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    if (lexical.DEPRECATED_$isGridSelection(selection)) {
      tableSelection.clearText();
      return true;
    } else if (lexical.$isRangeSelection(selection)) {
      const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }

      const anchorNode = selection.anchor.getNode();
      const focusNode = selection.focus.getNode();
      const isAnchorInside = tableNode.isParentOf(anchorNode);
      const isFocusInside = tableNode.isParentOf(focusNode);
      const selectionContainsPartialTable = isAnchorInside && !isFocusInside || isFocusInside && !isAnchorInside;

      if (selectionContainsPartialTable) {
        tableSelection.clearText();
        return true;
      }

      const nearestElementNode = utils.$findMatchingParent(selection.anchor.getNode(), n => lexical.$isElementNode(n));
      const topLevelCellElementNode = nearestElementNode && utils.$findMatchingParent(nearestElementNode, n => lexical.$isElementNode(n) && $isTableCellNode(n.getParent()));

      if (!lexical.$isElementNode(topLevelCellElementNode) || !lexical.$isElementNode(nearestElementNode)) {
        return false;
      }

      if (command === lexical.DELETE_LINE_COMMAND && topLevelCellElementNode.getPreviousSibling() === null) {
        // TODO: Fix Delete Line in Table Cells.
        return true;
      }

      if (command === lexical.DELETE_CHARACTER_COMMAND || command === lexical.DELETE_WORD_COMMAND) {
        if (selection.isCollapsed() && selection.anchor.offset === 0) {
          if (nearestElementNode !== topLevelCellElementNode) {
            const children = nearestElementNode.getChildren();
            const newParagraphNode = lexical.$createParagraphNode();
            children.forEach(child => newParagraphNode.append(child));
            nearestElementNode.replace(newParagraphNode);
            nearestElementNode.getWritable().__parent = tableCellNode.getKey();
            return true;
          }
        }
      }
    }

    return false;
  };

  [lexical.DELETE_WORD_COMMAND, lexical.DELETE_LINE_COMMAND, lexical.DELETE_CHARACTER_COMMAND].forEach(command => {
    tableSelection.listenersToRemove.add(editor.registerCommand(command, deleteTextHandler(command), lexical.COMMAND_PRIORITY_CRITICAL));
  });

  const deleteCellHandler = event => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    if (lexical.DEPRECATED_$isGridSelection(selection)) {
      event.preventDefault();
      event.stopPropagation();
      tableSelection.clearText();
      return true;
    } else if (lexical.$isRangeSelection(selection)) {
      const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }
    }

    return false;
  };

  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.KEY_BACKSPACE_COMMAND, deleteCellHandler, lexical.COMMAND_PRIORITY_CRITICAL));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.KEY_DELETE_COMMAND, deleteCellHandler, lexical.COMMAND_PRIORITY_CRITICAL));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.FORMAT_TEXT_COMMAND, payload => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    if (lexical.DEPRECATED_$isGridSelection(selection)) {
      tableSelection.formatCells(payload);
      return true;
    } else if (lexical.$isRangeSelection(selection)) {
      const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }
    }

    return false;
  }, lexical.COMMAND_PRIORITY_CRITICAL));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.CONTROLLED_TEXT_INSERTION_COMMAND, payload => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    if (lexical.DEPRECATED_$isGridSelection(selection)) {
      tableSelection.clearHighlight();
      return false;
    } else if (lexical.$isRangeSelection(selection)) {
      const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }
    }

    return false;
  }, lexical.COMMAND_PRIORITY_CRITICAL));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.KEY_TAB_COMMAND, event => {
    const selection = lexical.$getSelection();

    if (!$isSelectionInTable(selection, tableNode)) {
      return false;
    }

    if (lexical.$isRangeSelection(selection)) {
      const tableCellNode = utils.$findMatchingParent(selection.anchor.getNode(), n => $isTableCellNode(n));

      if (!$isTableCellNode(tableCellNode)) {
        return false;
      }

      if (selection.isCollapsed()) {
        const currentCords = tableNode.getCordsFromCellNode(tableCellNode, tableSelection.grid);
        event.preventDefault();
        selectGridNodeInDirection(tableSelection, tableNode, currentCords.x, currentCords.y, !event.shiftKey ? 'forward' : 'backward');
        return true;
      }
    }

    return false;
  }, lexical.COMMAND_PRIORITY_HIGH));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.FOCUS_COMMAND, payload => {
    return tableNode.isSelected();
  }, lexical.COMMAND_PRIORITY_HIGH));
  tableSelection.listenersToRemove.add(editor.registerCommand(lexical.SELECTION_CHANGE_COMMAND, payload => {
    const selection = lexical.$getSelection();
    const prevSelection = lexical.$getPreviousSelection();

    if (selection && lexical.$isRangeSelection(selection) && !selection.isCollapsed()) {
      const anchorNode = selection.anchor.getNode();
      const focusNode = selection.focus.getNode();
      const isAnchorInside = tableNode.isParentOf(anchorNode);
      const isFocusInside = tableNode.isParentOf(focusNode);
      const selectionContainsPartialTable = isAnchorInside && !isFocusInside || isFocusInside && !isAnchorInside;
      const selectionIsInsideTable = isAnchorInside && isFocusInside && !tableNode.isSelected();

      if (selectionContainsPartialTable) {
        const isBackward = selection.isBackward();
        const modifiedSelection = lexical.$createRangeSelection();
        const tableKey = tableNode.getKey();
        modifiedSelection.anchor.set(selection.anchor.key, selection.anchor.offset, selection.anchor.type);
        modifiedSelection.focus.set(tableKey, isBackward ? 0 : tableNode.getChildrenSize(), 'element');
        isRangeSelectionHijacked = true;
        lexical.$setSelection(modifiedSelection);
        $addHighlightStyleToTable(tableSelection);
        return true;
      } else if (selectionIsInsideTable) {
        const {
          grid
        } = tableSelection;

        if (selection.getNodes().filter($isTableCellNode).length === grid.rows * grid.columns) {
          const gridSelection = lexical.DEPRECATED_$createGridSelection();
          const tableKey = tableNode.getKey();
          const firstCell = tableNode.getFirstChildOrThrow().getFirstChild();
          const lastCell = tableNode.getLastChildOrThrow().getLastChild();

          if (firstCell != null && lastCell != null) {
            gridSelection.set(tableKey, firstCell.getKey(), lastCell.getKey());
            lexical.$setSelection(gridSelection);
            tableSelection.updateTableGridSelection(gridSelection);
            return true;
          }
        }
      }
    }

    if (selection && !selection.is(prevSelection) && (lexical.DEPRECATED_$isGridSelection(selection) || lexical.DEPRECATED_$isGridSelection(prevSelection)) && tableSelection.gridSelection && !tableSelection.gridSelection.is(prevSelection)) {
      if (lexical.DEPRECATED_$isGridSelection(selection) && selection.gridKey === tableSelection.tableNodeKey) {
        tableSelection.updateTableGridSelection(selection);
      } else if (!lexical.DEPRECATED_$isGridSelection(selection) && lexical.DEPRECATED_$isGridSelection(prevSelection) && prevSelection.gridKey === tableSelection.tableNodeKey) {
        tableSelection.updateTableGridSelection(null);
      }

      return false;
    }

    if (tableSelection.hasHijackedSelectionStyles && !tableNode.isSelected()) {
      $removeHighlightStyleToTable(tableSelection);
      isRangeSelectionHijacked = false;
    } else if (!tableSelection.hasHijackedSelectionStyles && tableNode.isSelected()) {
      $addHighlightStyleToTable(tableSelection);
    }

    return false;
  }, lexical.COMMAND_PRIORITY_CRITICAL));
  return tableSelection;
}
function attachTableSelectionToTableElement(tableElement, tableSelection) {
  tableElement[LEXICAL_ELEMENT_KEY] = tableSelection;
}
function getTableSelectionFromTableElement(tableElement) {
  return tableElement[LEXICAL_ELEMENT_KEY];
}
function getCellFromTarget(node) {
  let currentNode = node;

  while (currentNode != null) {
    const nodeName = currentNode.nodeName;

    if (nodeName === 'TD' || nodeName === 'TH') {
      // @ts-expect-error: internal field
      const cell = currentNode._cell;

      if (cell === undefined) {
        return null;
      }

      return cell;
    }

    currentNode = currentNode.parentNode;
  }

  return null;
}
function getTableGrid(tableElement) {
  const cells = [];
  const grid = {
    cells,
    columns: 0,
    rows: 0
  };
  let currentNode = tableElement.firstChild;
  let x = 0;
  let y = 0;
  cells.length = 0;

  while (currentNode != null) {
    const nodeMame = currentNode.nodeName;

    if (nodeMame === 'TD' || nodeMame === 'TH') {
      const elem = currentNode;
      const cell = {
        elem,
        highlighted: false,
        x,
        y
      }; // @ts-expect-error: internal field

      currentNode._cell = cell;

      if (cells[y] === undefined) {
        cells[y] = [];
      }

      cells[y][x] = cell;
    } else {
      const child = currentNode.firstChild;

      if (child != null) {
        currentNode = child;
        continue;
      }
    }

    const sibling = currentNode.nextSibling;

    if (sibling != null) {
      x++;
      currentNode = sibling;
      continue;
    }

    const parent = currentNode.parentNode;

    if (parent != null) {
      const parentSibling = parent.nextSibling;

      if (parentSibling == null) {
        break;
      }

      y++;
      x = 0;
      currentNode = parentSibling;
    }
  }

  grid.columns = x + 1;
  grid.rows = y + 1;
  return grid;
}
function $updateDOMForSelection(grid, selection) {
  const highlightedCells = [];
  const selectedCellNodes = new Set(selection ? selection.getNodes() : []);
  $forEachGridCell(grid, (cell, lexicalNode) => {
    const elem = cell.elem;

    if (selectedCellNodes.has(lexicalNode)) {
      cell.highlighted = true;
      elem.style.setProperty('background-color', 'rgb(172, 206, 247)');
      elem.style.setProperty('caret-color', 'transparent');
      highlightedCells.push(cell);
    } else {
      cell.highlighted = false;
      elem.style.removeProperty('background-color');
      elem.style.removeProperty('caret-color');

      if (!elem.getAttribute('style')) {
        elem.removeAttribute('style');
      }
    }
  });
  return highlightedCells;
}
function $forEachGridCell(grid, cb) {
  const {
    cells
  } = grid;

  for (let y = 0; y < cells.length; y++) {
    const row = cells[y];

    for (let x = 0; x < row.length; x++) {
      const cell = row[x];
      const lexicalNode = lexical.$getNearestNodeFromDOMNode(cell.elem);

      if (lexicalNode !== null) {
        cb(cell, lexicalNode, {
          x,
          y
        });
      }
    }
  }
}
function $addHighlightStyleToTable(tableSelection) {
  tableSelection.disableHighlightStyle();
  $forEachGridCell(tableSelection.grid, cell => {
    const elem = cell.elem;
    cell.highlighted = true;
    elem.style.setProperty('background-color', 'rgb(172, 206, 247)');
    elem.style.setProperty('caret-color', 'transparent');
  });
}
function $removeHighlightStyleToTable(tableSelection) {
  tableSelection.enableHighlightStyle();
  $forEachGridCell(tableSelection.grid, cell => {
    const elem = cell.elem;
    cell.highlighted = false;
    elem.style.removeProperty('background-color');
    elem.style.removeProperty('caret-color');

    if (!elem.getAttribute('style')) {
      elem.removeAttribute('style');
    }
  });
}

const selectGridNodeInDirection = (tableSelection, tableNode, x, y, direction) => {
  const isForward = direction === 'forward';

  switch (direction) {
    case 'backward':
    case 'forward':
      if (x !== (isForward ? tableSelection.grid.columns - 1 : 0)) {
        selectTableCellNode(tableNode.getCellNodeFromCordsOrThrow(x + (isForward ? 1 : -1), y, tableSelection.grid));
      } else {
        if (y !== (isForward ? tableSelection.grid.rows - 1 : 0)) {
          selectTableCellNode(tableNode.getCellNodeFromCordsOrThrow(isForward ? 0 : tableSelection.grid.columns - 1, y + (isForward ? 1 : -1), tableSelection.grid));
        } else if (!isForward) {
          tableNode.selectPrevious();
        } else {
          tableNode.selectNext();
        }
      }

      return true;

    case 'up':
      if (y !== 0) {
        selectTableCellNode(tableNode.getCellNodeFromCordsOrThrow(x, y - 1, tableSelection.grid));
      } else {
        tableNode.selectPrevious();
      }

      return true;

    case 'down':
      if (y !== tableSelection.grid.rows - 1) {
        selectTableCellNode(tableNode.getCellNodeFromCordsOrThrow(x, y + 1, tableSelection.grid));
      } else {
        tableNode.selectNext();
      }

      return true;

    default:
      return false;
  }
};

const adjustFocusNodeInDirection = (tableSelection, tableNode, x, y, direction) => {
  const isForward = direction === 'forward';

  switch (direction) {
    case 'backward':
    case 'forward':
      if (x !== (isForward ? tableSelection.grid.columns - 1 : 0)) {
        tableSelection.setFocusCellForSelection(tableNode.getCellFromCordsOrThrow(x + (isForward ? 1 : -1), y, tableSelection.grid));
      }

      return true;

    case 'up':
      if (y !== 0) {
        tableSelection.setFocusCellForSelection(tableNode.getCellFromCordsOrThrow(x, y - 1, tableSelection.grid));
        return true;
      } else {
        return false;
      }

    case 'down':
      if (y !== tableSelection.grid.rows - 1) {
        tableSelection.setFocusCellForSelection(tableNode.getCellFromCordsOrThrow(x, y + 1, tableSelection.grid));
        return true;
      } else {
        return false;
      }

    default:
      return false;
  }
};

function $isSelectionInTable(selection, tableNode) {
  if (lexical.$isRangeSelection(selection) || lexical.DEPRECATED_$isGridSelection(selection)) {
    const isAnchorInside = tableNode.isParentOf(selection.anchor.getNode());
    const isFocusInside = tableNode.isParentOf(selection.focus.getNode());
    return isAnchorInside && isFocusInside;
  }

  return false;
}

function selectTableCellNode(tableCell) {
  const possibleParagraph = tableCell.getChildren().find(n => lexical.$isParagraphNode(n));

  if (lexical.$isParagraphNode(possibleParagraph)) {
    possibleParagraph.selectEnd();
  } else {
    tableCell.selectEnd();
  }
}

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */

/** @noInheritDoc */
class TableNode extends lexical.DEPRECATED_GridNode {
  /** @internal */
  static getType() {
    return 'table';
  }

  static clone(node) {
    return new TableNode(node.__key);
  }

  static importDOM() {
    return {
      table: _node => ({
        conversion: convertTableElement,
        priority: 1
      })
    };
  }

  static importJSON(_serializedNode) {
    return $createTableNode();
  }

  constructor(key) {
    super(key);
  }

  exportJSON() {
    return { ...super.exportJSON(),
      type: 'table',
      version: 1
    };
  }

  createDOM(config, editor) {
    const tableElement = document.createElement('table');
    utils.addClassNamesToElement(tableElement, config.theme.table);
    return tableElement;
  }

  updateDOM() {
    return false;
  }

  exportDOM(editor) {
    return { ...super.exportDOM(editor),
      after: tableElement => {
        if (tableElement) {
          const newElement = tableElement.cloneNode();
          const colGroup = document.createElement('colgroup');
          const tBody = document.createElement('tbody');
          tBody.append(...tableElement.children);
          const firstRow = this.getFirstChildOrThrow();

          if (!$isTableRowNode(firstRow)) {
            throw new Error('Expected to find row node.');
          }

          const colCount = firstRow.getChildrenSize();

          for (let i = 0; i < colCount; i++) {
            const col = document.createElement('col');
            colGroup.append(col);
          }

          newElement.replaceChildren(colGroup, tBody);
          return newElement;
        }
      }
    };
  }

  canExtractContents() {
    return false;
  }

  canBeEmpty() {
    return false;
  }

  isShadowRoot() {
    return true;
  }

  getCordsFromCellNode(tableCellNode, grid) {
    const {
      rows,
      cells
    } = grid;

    for (let y = 0; y < rows; y++) {
      const row = cells[y];

      if (row == null) {
        throw new Error(`Row not found at y:${y}`);
      }

      const x = row.findIndex(({
        elem
      }) => {
        const cellNode = lexical.$getNearestNodeFromDOMNode(elem);
        return cellNode === tableCellNode;
      });

      if (x !== -1) {
        return {
          x,
          y
        };
      }
    }

    throw new Error('Cell not found in table.');
  }

  getCellFromCords(x, y, grid) {
    const {
      cells
    } = grid;
    const row = cells[y];

    if (row == null) {
      return null;
    }

    const cell = row[x];

    if (cell == null) {
      return null;
    }

    return cell;
  }

  getCellFromCordsOrThrow(x, y, grid) {
    const cell = this.getCellFromCords(x, y, grid);

    if (!cell) {
      throw new Error('Cell not found at cords.');
    }

    return cell;
  }

  getCellNodeFromCords(x, y, grid) {
    const cell = this.getCellFromCords(x, y, grid);

    if (cell == null) {
      return null;
    }

    const node = lexical.$getNearestNodeFromDOMNode(cell.elem);

    if ($isTableCellNode(node)) {
      return node;
    }

    return null;
  }

  getCellNodeFromCordsOrThrow(x, y, grid) {
    const node = this.getCellNodeFromCords(x, y, grid);

    if (!node) {
      throw new Error('Node at cords not TableCellNode.');
    }

    return node;
  }

  canSelectBefore() {
    return true;
  }

  canIndent() {
    return false;
  }

}
function $getElementGridForTableNode(editor, tableNode) {
  const tableElement = editor.getElementByKey(tableNode.getKey());

  if (tableElement == null) {
    throw new Error('Table Element Not Found');
  }

  return getTableGrid(tableElement);
}
function convertTableElement(_domNode) {
  return {
    node: $createTableNode()
  };
}
function $createTableNode() {
  return lexical.$applyNodeReplacement(new TableNode());
}
function $isTableNode(node) {
  return node instanceof TableNode;
}

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
function $createTableNodeWithDimensions(rowCount, columnCount, includeHeaders = true) {
  const tableNode = $createTableNode();

  for (let iRow = 0; iRow < rowCount; iRow++) {
    const tableRowNode = $createTableRowNode();

    for (let iColumn = 0; iColumn < columnCount; iColumn++) {
      let headerState = TableCellHeaderStates.NO_STATUS;

      if (includeHeaders) {
        if (iRow === 0) headerState |= TableCellHeaderStates.ROW;
        if (iColumn === 0) headerState |= TableCellHeaderStates.COLUMN;
      }

      const tableCellNode = $createTableCellNode(headerState);
      const paragraphNode = lexical.$createParagraphNode();
      paragraphNode.append(lexical.$createTextNode());
      tableCellNode.append(paragraphNode);
      tableRowNode.append(tableCellNode);
    }

    tableNode.append(tableRowNode);
  }

  return tableNode;
}
function $getTableCellNodeFromLexicalNode(startingNode) {
  const node = utils.$findMatchingParent(startingNode, n => $isTableCellNode(n));

  if ($isTableCellNode(node)) {
    return node;
  }

  return null;
}
function $getTableRowNodeFromTableCellNodeOrThrow(startingNode) {
  const node = utils.$findMatchingParent(startingNode, n => $isTableRowNode(n));

  if ($isTableRowNode(node)) {
    return node;
  }

  throw new Error('Expected table cell to be inside of table row.');
}
function $getTableNodeFromLexicalNodeOrThrow(startingNode) {
  const node = utils.$findMatchingParent(startingNode, n => $isTableNode(n));

  if ($isTableNode(node)) {
    return node;
  }

  throw new Error('Expected table cell to be inside of table.');
}
function $getTableRowIndexFromTableCellNode(tableCellNode) {
  const tableRowNode = $getTableRowNodeFromTableCellNodeOrThrow(tableCellNode);
  const tableNode = $getTableNodeFromLexicalNodeOrThrow(tableRowNode);
  return tableNode.getChildren().findIndex(n => n.is(tableRowNode));
}
function $getTableColumnIndexFromTableCellNode(tableCellNode) {
  const tableRowNode = $getTableRowNodeFromTableCellNodeOrThrow(tableCellNode);
  return tableRowNode.getChildren().findIndex(n => n.is(tableCellNode));
}
function $getTableCellSiblingsFromTableCellNode(tableCellNode, grid) {
  const tableNode = $getTableNodeFromLexicalNodeOrThrow(tableCellNode);
  const {
    x,
    y
  } = tableNode.getCordsFromCellNode(tableCellNode, grid);
  return {
    above: tableNode.getCellNodeFromCords(x, y - 1, grid),
    below: tableNode.getCellNodeFromCords(x, y + 1, grid),
    left: tableNode.getCellNodeFromCords(x - 1, y, grid),
    right: tableNode.getCellNodeFromCords(x + 1, y, grid)
  };
}
function $removeTableRowAtIndex(tableNode, indexToDelete) {
  const tableRows = tableNode.getChildren();

  if (indexToDelete >= tableRows.length || indexToDelete < 0) {
    throw new Error('Expected table cell to be inside of table row.');
  }

  const targetRowNode = tableRows[indexToDelete];
  targetRowNode.remove();
  return tableNode;
}
function $insertTableRow(tableNode, targetIndex, shouldInsertAfter = true, rowCount, grid) {
  const tableRows = tableNode.getChildren();

  if (targetIndex >= tableRows.length || targetIndex < 0) {
    throw new Error('Table row target index out of range');
  }

  const targetRowNode = tableRows[targetIndex];

  if ($isTableRowNode(targetRowNode)) {
    for (let r = 0; r < rowCount; r++) {
      const tableRowCells = targetRowNode.getChildren();
      const tableColumnCount = tableRowCells.length;
      const newTableRowNode = $createTableRowNode();

      for (let c = 0; c < tableColumnCount; c++) {
        const tableCellFromTargetRow = tableRowCells[c];

        if (!$isTableCellNode(tableCellFromTargetRow)) {
          throw Error(`Expected table cell`);
        }

        const {
          above,
          below
        } = $getTableCellSiblingsFromTableCellNode(tableCellFromTargetRow, grid);
        let headerState = TableCellHeaderStates.NO_STATUS;
        const width = above && above.getWidth() || below && below.getWidth() || undefined;

        if (above && above.hasHeaderState(TableCellHeaderStates.COLUMN) || below && below.hasHeaderState(TableCellHeaderStates.COLUMN)) {
          headerState |= TableCellHeaderStates.COLUMN;
        }

        const tableCellNode = $createTableCellNode(headerState, 1, width);
        tableCellNode.append(lexical.$createParagraphNode());
        newTableRowNode.append(tableCellNode);
      }

      if (shouldInsertAfter) {
        targetRowNode.insertAfter(newTableRowNode);
      } else {
        targetRowNode.insertBefore(newTableRowNode);
      }
    }
  } else {
    throw new Error('Row before insertion index does not exist.');
  }

  return tableNode;
}
function $insertTableColumn(tableNode, targetIndex, shouldInsertAfter = true, columnCount, grid) {
  const tableRows = tableNode.getChildren();

  for (let r = 0; r < tableRows.length; r++) {
    const currentTableRowNode = tableRows[r];

    if ($isTableRowNode(currentTableRowNode)) {
      for (let c = 0; c < columnCount; c++) {
        const tableRowChildren = currentTableRowNode.getChildren();

        if (targetIndex >= tableRowChildren.length || targetIndex < 0) {
          throw new Error('Table column target index out of range');
        }

        const targetCell = tableRowChildren[targetIndex];

        if (!$isTableCellNode(targetCell)) {
          throw Error(`Expected table cell`);
        }

        const {
          left,
          right
        } = $getTableCellSiblingsFromTableCellNode(targetCell, grid);
        let headerState = TableCellHeaderStates.NO_STATUS;

        if (left && left.hasHeaderState(TableCellHeaderStates.ROW) || right && right.hasHeaderState(TableCellHeaderStates.ROW)) {
          headerState |= TableCellHeaderStates.ROW;
        }

        const newTableCell = $createTableCellNode(headerState);
        newTableCell.append(lexical.$createParagraphNode());

        if (shouldInsertAfter) {
          targetCell.insertAfter(newTableCell);
        } else {
          targetCell.insertBefore(newTableCell);
        }
      }
    }
  }

  return tableNode;
}
function $deleteTableColumn(tableNode, targetIndex) {
  const tableRows = tableNode.getChildren();

  for (let i = 0; i < tableRows.length; i++) {
    const currentTableRowNode = tableRows[i];

    if ($isTableRowNode(currentTableRowNode)) {
      const tableRowChildren = currentTableRowNode.getChildren();

      if (targetIndex >= tableRowChildren.length || targetIndex < 0) {
        throw new Error('Table column target index out of range');
      }

      tableRowChildren[targetIndex].remove();
    }
  }

  return tableNode;
}

/** @module @lexical/table */
const INSERT_TABLE_COMMAND = lexical.createCommand('INSERT_TABLE_COMMAND');

exports.$createTableCellNode = $createTableCellNode;
exports.$createTableNode = $createTableNode;
exports.$createTableNodeWithDimensions = $createTableNodeWithDimensions;
exports.$createTableRowNode = $createTableRowNode;
exports.$deleteTableColumn = $deleteTableColumn;
exports.$getElementGridForTableNode = $getElementGridForTableNode;
exports.$getTableCellNodeFromLexicalNode = $getTableCellNodeFromLexicalNode;
exports.$getTableColumnIndexFromTableCellNode = $getTableColumnIndexFromTableCellNode;
exports.$getTableNodeFromLexicalNodeOrThrow = $getTableNodeFromLexicalNodeOrThrow;
exports.$getTableRowIndexFromTableCellNode = $getTableRowIndexFromTableCellNode;
exports.$getTableRowNodeFromTableCellNodeOrThrow = $getTableRowNodeFromTableCellNodeOrThrow;
exports.$insertTableColumn = $insertTableColumn;
exports.$insertTableRow = $insertTableRow;
exports.$isTableCellNode = $isTableCellNode;
exports.$isTableNode = $isTableNode;
exports.$isTableRowNode = $isTableRowNode;
exports.$removeTableRowAtIndex = $removeTableRowAtIndex;
exports.INSERT_TABLE_COMMAND = INSERT_TABLE_COMMAND;
exports.TableCellHeaderStates = TableCellHeaderStates;
exports.TableCellNode = TableCellNode;
exports.TableNode = TableNode;
exports.TableRowNode = TableRowNode;
exports.TableSelection = TableSelection;
exports.applyTableHandlers = applyTableHandlers;
exports.getCellFromTarget = getCellFromTarget;
exports.getTableSelectionFromTableElement = getTableSelectionFromTableElement;
