/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
'use strict';

var lexical = require('lexical');

/** @module @lexical/utils */
function addClassNamesToElement(element, ...classNames) {
  classNames.forEach(className => {
    if (typeof className === 'string') {
      const classesToAdd = className.split(' ').filter(n => n !== '');
      element.classList.add(...classesToAdd);
    }
  });
}
function removeClassNamesFromElement(element, ...classNames) {
  classNames.forEach(className => {
    if (typeof className === 'string') {
      element.classList.remove(...className.split(' '));
    }
  });
}
function isMimeType(file, acceptableMimeTypes) {
  for (const acceptableType of acceptableMimeTypes) {
    if (file.type.startsWith(acceptableType)) {
      return true;
    }
  }

  return false;
}
/**
 * Lexical File Reader with:
 *  1. MIME type support
 *  2. batched results (HistoryPlugin compatibility)
 *  3. Order aware (respects the order when multiple Files are passed)
 *
 * const filesResult = await mediaFileReader(files, ['image/']);
 * filesResult.forEach(file => editor.dispatchCommand('INSERT_IMAGE', {
 *   src: file.result,
 * }));
 */

function mediaFileReader(files, acceptableMimeTypes) {
  const filesIterator = files[Symbol.iterator]();
  return new Promise((resolve, reject) => {
    const processed = [];

    const handleNextFile = () => {
      const {
        done,
        value: file
      } = filesIterator.next();

      if (done) {
        return resolve(processed);
      }

      const fileReader = new FileReader();
      fileReader.addEventListener('error', reject);
      fileReader.addEventListener('load', () => {
        const result = fileReader.result;

        if (typeof result === 'string') {
          processed.push({
            file,
            result
          });
        }

        handleNextFile();
      });

      if (isMimeType(file, acceptableMimeTypes)) {
        fileReader.readAsDataURL(file);
      } else {
        handleNextFile();
      }
    };

    handleNextFile();
  });
}
function $dfs(startingNode, endingNode) {
  const nodes = [];
  const start = (startingNode || lexical.$getRoot()).getLatest();
  const end = endingNode || (lexical.$isElementNode(start) ? start.getLastDescendant() : start);
  let node = start;
  let depth = $getDepth(node);

  while (node !== null && !node.is(end)) {
    nodes.push({
      depth,
      node
    });

    if (lexical.$isElementNode(node) && node.getChildrenSize() > 0) {
      node = node.getFirstChild();
      depth++;
    } else {
      // Find immediate sibling or nearest parent sibling
      let sibling = null;

      while (sibling === null && node !== null) {
        sibling = node.getNextSibling();

        if (sibling === null) {
          node = node.getParent();
          depth--;
        } else {
          node = sibling;
        }
      }
    }
  }

  if (node !== null && node.is(end)) {
    nodes.push({
      depth,
      node
    });
  }

  return nodes;
}

function $getDepth(node) {
  let innerNode = node;
  let depth = 0;

  while ((innerNode = innerNode.getParent()) !== null) {
    depth++;
  }

  return depth;
}

function $getNearestNodeOfType(node, klass) {
  let parent = node;

  while (parent != null) {
    if (parent instanceof klass) {
      return parent;
    }

    parent = parent.getParent();
  }

  return null;
}
function $getNearestBlockElementAncestorOrThrow(startNode) {
  const blockNode = $findMatchingParent(startNode, node => lexical.$isElementNode(node) && !node.isInline());

  if (!lexical.$isElementNode(blockNode)) {
    {
      throw Error(`Expected node ${startNode.__key} to have closest block element node.`);
    }
  }

  return blockNode;
}
function $findMatchingParent(startingNode, findFn) {
  let curr = startingNode;

  while (curr !== lexical.$getRoot() && curr != null) {
    if (findFn(curr)) {
      return curr;
    }

    curr = curr.getParent();
  }

  return null;
}
function mergeRegister(...func) {
  return () => {
    func.forEach(f => f());
  };
}
function registerNestedElementResolver(editor, targetNode, cloneNode, handleOverlap) {
  const $isTargetNode = node => {
    return node instanceof targetNode;
  };

  const $findMatch = node => {
    // First validate we don't have any children that are of the target,
    // as we need to handle them first.
    const children = node.getChildren();

    for (let i = 0; i < children.length; i++) {
      const child = children[i];

      if ($isTargetNode(child)) {
        return null;
      }
    }

    let parentNode = node;
    let childNode = node;

    while (parentNode !== null) {
      childNode = parentNode;
      parentNode = parentNode.getParent();

      if ($isTargetNode(parentNode)) {
        return {
          child: childNode,
          parent: parentNode
        };
      }
    }

    return null;
  };

  const elementNodeTransform = node => {
    const match = $findMatch(node);

    if (match !== null) {
      const {
        child,
        parent
      } = match; // Simple path, we can move child out and siblings into a new parent.

      if (child.is(node)) {
        handleOverlap(parent, node);
        const nextSiblings = child.getNextSiblings();
        const nextSiblingsLength = nextSiblings.length;
        parent.insertAfter(child);

        if (nextSiblingsLength !== 0) {
          const newParent = cloneNode(parent);
          child.insertAfter(newParent);

          for (let i = 0; i < nextSiblingsLength; i++) {
            newParent.append(nextSiblings[i]);
          }
        }

        if (!parent.canBeEmpty() && parent.getChildrenSize() === 0) {
          parent.remove();
        }
      }
    }
  };

  return editor.registerNodeTransform(targetNode, elementNodeTransform);
}
function $restoreEditorState(editor, editorState) {
  const FULL_RECONCILE = 2;
  const nodeMap = new Map(editorState._nodeMap);
  const activeEditorState = editor._pendingEditorState;

  if (activeEditorState) {
    activeEditorState._nodeMap = nodeMap;
  }

  editor._dirtyType = FULL_RECONCILE;
  const selection = editorState._selection;
  lexical.$setSelection(selection === null ? null : selection.clone());
}
function $insertNodeToNearestRoot(node) {
  const selection = lexical.$getSelection();

  if (lexical.$isRangeSelection(selection)) {
    const {
      focus
    } = selection;
    const focusNode = focus.getNode();
    const focusOffset = focus.offset;

    if (lexical.$isRootOrShadowRoot(focusNode)) {
      const focusChild = focusNode.getChildAtIndex(focusOffset);

      if (focusChild == null) {
        focusNode.append(node);
      } else {
        focusChild.insertBefore(node);
      }

      node.selectNext();
    } else {
      let splitNode;
      let splitOffset;

      if (lexical.$isTextNode(focusNode)) {
        splitNode = focusNode.getParentOrThrow();
        splitOffset = focusNode.getIndexWithinParent();

        if (focusOffset > 0) {
          splitOffset += 1;
          focusNode.splitText(focusOffset);
        }
      } else {
        splitNode = focusNode;
        splitOffset = focusOffset;
      }

      const [, rightTree] = $splitNode(splitNode, splitOffset);
      rightTree.insertBefore(node);
      rightTree.selectStart();
    }
  } else {
    if (lexical.$isNodeSelection(selection) || lexical.DEPRECATED_$isGridSelection(selection)) {
      const nodes = selection.getNodes();
      nodes[nodes.length - 1].getTopLevelElementOrThrow().insertAfter(node);
    } else {
      const root = lexical.$getRoot();
      root.append(node);
    }

    const paragraphNode = lexical.$createParagraphNode();
    node.insertAfter(paragraphNode);
    paragraphNode.select();
  }

  return node.getLatest();
}
function $wrapNodeInElement(node, createElementNode) {
  const elementNode = createElementNode();
  node.replace(elementNode);
  elementNode.append(node);
  return elementNode;
}
function $splitNode(node, offset) {
  let startNode = node.getChildAtIndex(offset);

  if (startNode == null) {
    startNode = node;
  }

  if (!!lexical.$isRootOrShadowRoot(node)) {
    throw Error(`Can not call $splitNode() on root element`);
  }

  const recurse = currentNode => {
    const parent = currentNode.getParentOrThrow();
    const isParentRoot = lexical.$isRootOrShadowRoot(parent); // The node we start split from (leaf) is moved, but its recursive
    // parents are copied to create separate tree

    const nodeToMove = currentNode === startNode && !isParentRoot ? currentNode : lexical.$copyNode(currentNode);

    if (isParentRoot) {
      currentNode.insertAfter(nodeToMove);
      return [currentNode, nodeToMove, nodeToMove];
    } else {
      const [leftTree, rightTree, newParent] = recurse(parent);
      const nextSiblings = currentNode.getNextSiblings();
      newParent.append(nodeToMove, ...nextSiblings);
      return [leftTree, rightTree, nodeToMove];
    }
  };

  const [leftTree, rightTree] = recurse(startNode);
  return [leftTree, rightTree];
}

exports.$dfs = $dfs;
exports.$findMatchingParent = $findMatchingParent;
exports.$getNearestBlockElementAncestorOrThrow = $getNearestBlockElementAncestorOrThrow;
exports.$getNearestNodeOfType = $getNearestNodeOfType;
exports.$insertNodeToNearestRoot = $insertNodeToNearestRoot;
exports.$restoreEditorState = $restoreEditorState;
exports.$splitNode = $splitNode;
exports.$wrapNodeInElement = $wrapNodeInElement;
exports.addClassNamesToElement = addClassNamesToElement;
exports.isMimeType = isMimeType;
exports.mediaFileReader = mediaFileReader;
exports.mergeRegister = mergeRegister;
exports.registerNestedElementResolver = registerNestedElementResolver;
exports.removeClassNamesFromElement = removeClassNamesFromElement;
