/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
import type { EditorState, SerializedEditorState } from './LexicalEditorState';
import type { DOMConversion, NodeKey } from './LexicalNode';
import { LexicalNode } from './LexicalNode';
export declare type Spread<T1, T2> = Omit<T2, keyof T1> & T1;
export declare type Klass<T extends LexicalNode> = {
    new (...args: any[]): T;
} & Omit<LexicalNode, 'constructor'>;
export declare type EditorThemeClassName = string;
export declare type TextNodeThemeClasses = {
    base?: EditorThemeClassName;
    bold?: EditorThemeClassName;
    code?: EditorThemeClassName;
    italic?: EditorThemeClassName;
    strikethrough?: EditorThemeClassName;
    subscript?: EditorThemeClassName;
    superscript?: EditorThemeClassName;
    underline?: EditorThemeClassName;
    underlineStrikethrough?: EditorThemeClassName;
};
export declare type EditorUpdateOptions = {
    onUpdate?: () => void;
    skipTransforms?: true;
    tag?: string;
    discrete?: true;
};
export declare type EditorSetOptions = {
    tag?: string;
};
export declare type EditorFocusOptions = {
    defaultSelection?: 'rootStart' | 'rootEnd';
};
export declare type EditorThemeClasses = {
    blockCursor?: EditorThemeClassName;
    characterLimit?: EditorThemeClassName;
    code?: EditorThemeClassName;
    codeHighlight?: Record<string, EditorThemeClassName>;
    hashtag?: EditorThemeClassName;
    heading?: {
        h1?: EditorThemeClassName;
        h2?: EditorThemeClassName;
        h3?: EditorThemeClassName;
        h4?: EditorThemeClassName;
        h5?: EditorThemeClassName;
        h6?: EditorThemeClassName;
    };
    image?: EditorThemeClassName;
    link?: EditorThemeClassName;
    list?: {
        ul?: EditorThemeClassName;
        ulDepth?: Array<EditorThemeClassName>;
        ol?: EditorThemeClassName;
        olDepth?: Array<EditorThemeClassName>;
        listitem?: EditorThemeClassName;
        listitemChecked?: EditorThemeClassName;
        listitemUnchecked?: EditorThemeClassName;
        nested?: {
            list?: EditorThemeClassName;
            listitem?: EditorThemeClassName;
        };
    };
    ltr?: EditorThemeClassName;
    mark?: EditorThemeClassName;
    markOverlap?: EditorThemeClassName;
    paragraph?: EditorThemeClassName;
    quote?: EditorThemeClassName;
    root?: EditorThemeClassName;
    rtl?: EditorThemeClassName;
    table?: EditorThemeClassName;
    tableAddColumns?: EditorThemeClassName;
    tableAddRows?: EditorThemeClassName;
    tableCellActionButton?: EditorThemeClassName;
    tableCellActionButtonContainer?: EditorThemeClassName;
    tableCellPrimarySelected?: EditorThemeClassName;
    tableCellSelected?: EditorThemeClassName;
    tableCell?: EditorThemeClassName;
    tableCellEditing?: EditorThemeClassName;
    tableCellHeader?: EditorThemeClassName;
    tableCellResizer?: EditorThemeClassName;
    tableCellSortedIndicator?: EditorThemeClassName;
    tableResizeRuler?: EditorThemeClassName;
    tableRow?: EditorThemeClassName;
    tableSelected?: EditorThemeClassName;
    text?: TextNodeThemeClasses;
    embedBlock?: {
        base?: EditorThemeClassName;
        focus?: EditorThemeClassName;
    };
    [key: string]: any;
};
export declare type EditorConfig = {
    disableEvents?: boolean;
    namespace: string;
    theme: EditorThemeClasses;
};
export declare type RegisteredNodes = Map<string, RegisteredNode>;
export declare type RegisteredNode = {
    klass: Klass<LexicalNode>;
    transforms: Set<Transform<LexicalNode>>;
    replace: null | ((node: LexicalNode) => LexicalNode);
};
export declare type Transform<T extends LexicalNode> = (node: T) => void;
export declare type ErrorHandler = (error: Error) => void;
export declare type MutationListeners = Map<MutationListener, Klass<LexicalNode>>;
export declare type MutatedNodes = Map<Klass<LexicalNode>, Map<NodeKey, NodeMutation>>;
export declare type NodeMutation = 'created' | 'updated' | 'destroyed';
export declare type UpdateListener = (arg0: {
    dirtyElements: Map<NodeKey, IntentionallyMarkedAsDirtyElement>;
    dirtyLeaves: Set<NodeKey>;
    editorState: EditorState;
    normalizedNodes: Set<NodeKey>;
    prevEditorState: EditorState;
    tags: Set<string>;
}) => void;
export declare type DecoratorListener<T = never> = (decorator: Record<NodeKey, T>) => void;
export declare type RootListener = (rootElement: null | HTMLElement, prevRootElement: null | HTMLElement) => void;
export declare type TextContentListener = (text: string) => void;
export declare type MutationListener = (nodes: Map<NodeKey, NodeMutation>, payload: {
    updateTags: Set<string>;
    dirtyLeaves: Set<string>;
}) => void;
export declare type CommandListener<P> = (payload: P, editor: LexicalEditor) => boolean;
export declare type EditableListener = (editable: boolean) => void;
export declare type CommandListenerPriority = 0 | 1 | 2 | 3 | 4;
export declare const COMMAND_PRIORITY_EDITOR = 0;
export declare const COMMAND_PRIORITY_LOW = 1;
export declare const COMMAND_PRIORITY_NORMAL = 2;
export declare const COMMAND_PRIORITY_HIGH = 3;
export declare const COMMAND_PRIORITY_CRITICAL = 4;
export declare type LexicalCommand<TPayload> = {
    type?: string;
};
/**
 * Type helper for extracting the payload type from a command.
 *
 * @example
 * ```ts
 * const MY_COMMAND = createCommand<SomeType>();
 *
 * // ...
 *
 * editor.registerCommand(MY_COMMAND, payload => {
 *   // Type of `payload` is inferred here. But lets say we want to extract a function to delegate to
 *   handleMyCommand(editor, payload);
 *   return true;
 * });
 *
 * function handleMyCommand(editor: LexicalEditor, payload: CommandPayloadType<typeof MY_COMMAND>) {
 *   // `payload` is of type `SomeType`, extracted from the command.
 * }
 * ```
 */
export declare type CommandPayloadType<TCommand extends LexicalCommand<unknown>> = TCommand extends LexicalCommand<infer TPayload> ? TPayload : never;
declare type Commands = Map<LexicalCommand<unknown>, Array<Set<CommandListener<unknown>>>>;
declare type Listeners = {
    decorator: Set<DecoratorListener>;
    mutation: MutationListeners;
    editable: Set<EditableListener>;
    root: Set<RootListener>;
    textcontent: Set<TextContentListener>;
    update: Set<UpdateListener>;
};
export declare type Listener = DecoratorListener | EditableListener | MutationListener | RootListener | TextContentListener | UpdateListener;
export declare type ListenerType = 'update' | 'root' | 'decorator' | 'textcontent' | 'mutation' | 'editable';
export declare type TransformerType = 'text' | 'decorator' | 'element' | 'root';
declare type IntentionallyMarkedAsDirtyElement = boolean;
declare type DOMConversionCache = Map<string, Array<(node: Node) => DOMConversion | null>>;
export declare type SerializedEditor = {
    editorState: SerializedEditorState;
};
export declare function resetEditor(editor: LexicalEditor, prevRootElement: null | HTMLElement, nextRootElement: null | HTMLElement, pendingEditorState: EditorState): void;
export declare function createEditor(editorConfig?: {
    disableEvents?: boolean;
    editorState?: EditorState;
    namespace?: string;
    nodes?: ReadonlyArray<Klass<LexicalNode> | {
        replace: Klass<LexicalNode>;
        with: <T extends {
            new (...args: any): any;
        }>(node: InstanceType<T>) => LexicalNode;
    }>;
    onError?: ErrorHandler;
    parentEditor?: LexicalEditor;
    editable?: boolean;
    theme?: EditorThemeClasses;
}): LexicalEditor;
export declare class LexicalEditor {
    _headless: boolean;
    _parentEditor: null | LexicalEditor;
    _rootElement: null | HTMLElement;
    _editorState: EditorState;
    _pendingEditorState: null | EditorState;
    _compositionKey: null | NodeKey;
    _deferred: Array<() => void>;
    _keyToDOMMap: Map<NodeKey, HTMLElement>;
    _updates: Array<[() => void, EditorUpdateOptions | undefined]>;
    _updating: boolean;
    _listeners: Listeners;
    _commands: Commands;
    _nodes: RegisteredNodes;
    _decorators: Record<NodeKey, unknown>;
    _pendingDecorators: null | Record<NodeKey, unknown>;
    _config: EditorConfig;
    _dirtyType: 0 | 1 | 2;
    _cloneNotNeeded: Set<NodeKey>;
    _dirtyLeaves: Set<NodeKey>;
    _dirtyElements: Map<NodeKey, IntentionallyMarkedAsDirtyElement>;
    _normalizedNodes: Set<NodeKey>;
    _updateTags: Set<string>;
    _observer: null | MutationObserver;
    _key: string;
    _onError: ErrorHandler;
    _htmlConversions: DOMConversionCache;
    _window: null | Window;
    _editable: boolean;
    _blockCursorElement: null | HTMLDivElement;
    constructor(editorState: EditorState, parentEditor: null | LexicalEditor, nodes: RegisteredNodes, config: EditorConfig, onError: ErrorHandler, htmlConversions: DOMConversionCache, editable: boolean);
    isComposing(): boolean;
    registerUpdateListener(listener: UpdateListener): () => void;
    registerEditableListener(listener: EditableListener): () => void;
    registerDecoratorListener<T>(listener: DecoratorListener<T>): () => void;
    registerTextContentListener(listener: TextContentListener): () => void;
    registerRootListener(listener: RootListener): () => void;
    registerCommand<P>(command: LexicalCommand<P>, listener: CommandListener<P>, priority: CommandListenerPriority): () => void;
    registerMutationListener(klass: Klass<LexicalNode>, listener: MutationListener): () => void;
    registerNodeTransform<T extends LexicalNode>(klass: Klass<T>, listener: Transform<T>): () => void;
    hasNodes<T extends Klass<LexicalNode>>(nodes: Array<T>): boolean;
    dispatchCommand<TCommand extends LexicalCommand<unknown>>(type: TCommand, payload: CommandPayloadType<TCommand>): boolean;
    getDecorators<T>(): Record<NodeKey, T>;
    getRootElement(): null | HTMLElement;
    getKey(): string;
    setRootElement(nextRootElement: null | HTMLElement): void;
    getElementByKey(key: NodeKey): HTMLElement | null;
    getEditorState(): EditorState;
    setEditorState(editorState: EditorState, options?: EditorSetOptions): void;
    parseEditorState(maybeStringifiedEditorState: string | SerializedEditorState, updateFn?: () => void): EditorState;
    update(updateFn: () => void, options?: EditorUpdateOptions): void;
    focus(callbackFn?: () => void, options?: EditorFocusOptions): void;
    blur(): void;
    isEditable(): boolean;
    setEditable(editable: boolean): void;
    toJSON(): SerializedEditor;
}
export {};
